package ru.yandex.mail.hackathon.todolistmodifier;

import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.mail.hackathon.CalendarFacade;
import ru.yandex.mail.hackathon.userrequest.UserRequest;
import ru.yandex.mail.hackathon.userrequest.UserRequestParser;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class TodoListModifier {
    private final Logger logger;

    private final CalendarFacade calendarFacade;

    private UserRequestParser parser = new UserRequestParser();

    public TodoListModifier(Logger logger, CalendarFacade calendarFacade) {
        this.logger = logger;
        this.calendarFacade = calendarFacade;
    }

    public TodoListModifyState process(long uid, String userMessage, String stateJson) {
        TodoListModifyState state = parseOrCreateState(stateJson);
        state.processingResult = TodoListModifyProcessingResult.OK;

        UserRequest userRequest;
        try {
            userRequest = parser.parseUserRequest(userMessage);
        } catch (IllegalArgumentException ex) {
            logger.log(Level.WARNING, "Could not parse user#" + uid + " message = " + userMessage, ex);
            state.processingResult = TodoListModifyProcessingResult.UNKNOWN_INPUT;
            return state;
        }

        try {
            return new RequestHandler(uid, userRequest, state).process();
        } catch (RuntimeException ex) {
            logger.log(Level.WARNING, "Error while processing user#" + uid +" message = " + userMessage, ex);
            return fail(new TodoListModifyState());
        }
    }

    private static TodoListModifyState parseOrCreateState(String currentState) {
        if (StringUtils.isBlank(currentState)) {
            return new TodoListModifyState();
        } else {
            return TodoListModifyState.parseJson(currentState);
        }
    }

    private static TodoListModifyState fail(TodoListModifyState state) {
        state.processingResult = TodoListModifyProcessingResult.ERROR;
        return state;
    }

    private class RequestHandler {
        private final long uid;

        private final UserRequest userRequest;

        private final TodoListModifyState state;

        private RequestHandler(long uid, UserRequest userRequest, TodoListModifyState state) {
            this.uid = uid;
            this.userRequest = userRequest;
            this.state = state;
        }

        private TodoListModifyState fail() {
            return TodoListModifier.fail(state);
        }

        private TodoListModifyState process() {
            if (userRequest.getIntent().isInitial && state.status != TodoListModifyStatus.START) {
                return fail();
            }

            if (userRequest.getIntent().hasListName) {
                if (userRequest.getListName().isEmpty()) {
                    return fail();
                }

                state.setListName(userRequest.getListName().get());
            }

            if (userRequest.getIntent().hasListItem) {
                if (userRequest.getItemNames().isEmpty()) {
                    return fail();
                }

                state.items = Cf.toList(userRequest.getItemNames());
            }

            Optional<String> listExternalId;
            switch (userRequest.getIntent()) {
                case CREATE_LIST:
                    listExternalId = calendarFacade.getTodoList(uid, state.getListName());
                    if (listExternalId.isPresent()) {
                        return changeStatus(TodoListModifyStatus.LIST_ALREADY_EXISTS);
                    }

                    createListAndSetExternalId();
                    return changeStatus(TodoListModifyStatus.LIST_CREATED);

                case ADD_ITEMS_TO_LIST:
                    listExternalId = calendarFacade.getTodoList(uid, state.getListName());
                    if (listExternalId.isEmpty()) {
                        return changeStatus(TodoListModifyStatus.LIST_IS_MISSING_WHEN_ADDING_ITEM);
                    }

                    return addItemsAndChangeStatus();

                case REMOVE_ITEM_FROM_LIST:
                    return fail();

                case SEND_LIST:
                    return fail();

                case CONFIRM:
                    if (state.status == TodoListModifyStatus.LIST_IS_MISSING_WHEN_ADDING_ITEM) {
                        createListAndSetExternalId();
                        return addItemsAndChangeStatus();
                    }
                    return fail();

                case DECLINE:
                    if (state.status == TodoListModifyStatus.LIST_IS_MISSING_WHEN_ADDING_ITEM) {
                        return changeStatus(TodoListModifyStatus.SILENT_OK);
                    }
                    return fail();
            }
            return fail();
        }

        private void createListAndSetExternalId() {
            String externalId = calendarFacade.createTodoList(uid, state.getListName());
            state.setListExternalId(externalId);
        }

        private TodoListModifyState addItemsAndChangeStatus() {
            for (String itemName : state.items) {
                calendarFacade.createTodoItem(uid, state.getListExternalId(), itemName);
            }
            state.status = TodoListModifyStatus.ITEMS_ADDED;
            return state;
        }

        private TodoListModifyState changeStatus(TodoListModifyStatus status) {
            state.status = status;
            return state;
        }
    }
}
