package ru.yandex.mail.hackathon.todolistmodifier;

import java.util.Optional;
import java.util.logging.Logger;

import org.junit.Assert;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.mail.hackathon.CalendarFacade;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class TodoListModifierTest {
    @Test
    public void testCreate() {
        assertResponseOnRequest(
                "Алиса, создай список покупок",
                "Я создала список покупок"
        );
    }

    @Test
    public void testCreateWhenExists() {
        assertResponseOnRequest(
                "Алиса, создай список покупок",
                "У вас уже есть список покупок",
                null,
                new CalendarFacadeMock()
                        .withExternalId("123")
        );
    }

    @Test
    public void testBrokenFacade() {
        assertResponseOnRequest(
                "Алиса, создай список покупок",
                "Что-то пошло не так",
                null,
                new CalendarFacadeMock()
                        .broke()
        );
    }

    @Test
    public void testUnknownInput() {
        assertResponseOnRequest(
                "Чо-как?",
                TodoListModifyState.UNKNOWN_INPUT_RESPONSE,
                null,
                new CalendarFacadeMock()
        );
    }

    @Test
    public void testAddWhenMissing() {
        TodoListModifyState state = assertResponseOnRequest(
                "Алиса, добавь в список покупок туалетную бумагу",
                "Я не нашла список покупок. Создать?"
        );
        assertResponseOnRequest(
                "Да",
                "Я добавила туалетную бумагу в список покупок",
                state.toJsonString()
        );
    }

    private TodoListModifyState assertResponseOnRequest(String request, String response) {
        return assertResponseOnRequest(request, response, null);
    }

    private TodoListModifyState assertResponseOnRequest(String request, String response, String stateJson) {
        return assertResponseOnRequest(request, response, stateJson, new CalendarFacadeMock());
    }

    private TodoListModifyState assertResponseOnRequest(String request, String response, String stateJson,
            CalendarFacadeMock calendarFacade) {
        TodoListModifyState state = new TodoListModifier(
                Mockito.mock(Logger.class),
                calendarFacade
        ).process(1, request, stateJson);
        Assert.assertEquals(
                response,
                state
                        .getResponseToUser()
        );
        return state;
    }

    private static class CalendarFacadeMock implements CalendarFacade {
        boolean broken = false;

        Optional<String> externalId = Optional.empty();

        CalendarFacadeMock withExternalId(String externalId) {
            this.externalId = Optional.of(externalId);
            return this;
        }

        @Override
        public Optional<String> getTodoList(long uid, String title) {
            if (broken) {
                throw new RuntimeException();
            }

            return externalId;
        }

        @Override
        public String createTodoList(long uid, String title) {
            if (broken) {
                throw new RuntimeException();
            }

            return null;
        }

        @Override
        public long createTodoItem(long uid, String listExternalId, String title) {
            if (broken) {
                throw new RuntimeException();
            }

            return 0;
        }

        public CalendarFacadeMock broke() {
            broken = true;
            return this;
        }
    }
}
