package ru.yandex.mail.hackathon.userrequest;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import ru.yandex.lemmer.Lemmer;

public class UserRequestParser {

    private static final Map<String, UserRequest.Intent> INTENTS_MAP = createIntendsMap();

    private static Map<String, UserRequest.Intent> createIntendsMap() {
        Map<String, UserRequest.Intent> map = new HashMap<>();
        map.put("добавь", UserRequest.Intent.ADD_ITEMS_TO_LIST);
        map.put("создай", UserRequest.Intent.CREATE_LIST);
        map.put("удали", UserRequest.Intent.REMOVE_ITEM_FROM_LIST);
        map.put("убери", UserRequest.Intent.REMOVE_ITEM_FROM_LIST);
        map.put("отправь", UserRequest.Intent.SEND_LIST);
        map.put("да", UserRequest.Intent.CONFIRM);
        map.put("нет", UserRequest.Intent.DECLINE);
        return map;
    }

    public UserRequest parseUserRequest(String input) {
        if (input == null || input.isBlank()) {
            throw new IllegalArgumentException("Empty string");
        }
        List<String> tokens = new ArrayList<>(Arrays.asList(input.toLowerCase().split("[\\s,\\.!\\?\\:;]+")));
        if ("алиса".equals(getFirst(tokens))) {
            deleteFirst(tokens);
        }
        if (tokens.isEmpty()) {
            throw new IllegalArgumentException("Empty command");
        }
        String actionToken = getFirst(tokens);
        UserRequest.Intent intent = INTENTS_MAP.get(actionToken);
        if (intent == null) {
            throw new IllegalArgumentException("Unknown intent");
        }
        deleteFirst(tokens);
        if (tokens.size() < intent.getMinTokensCountAfter()) {
            throw new IllegalArgumentException("Too short");
        }
        if (intent.getMinTokensCountAfter() == 0) {
            return new UserRequest(intent, Optional.empty(), Collections.emptyList());
        }
        if (intent.getPreposition().isPresent()) {
            if (!intent.getPreposition().map(preposition -> preposition.equals(getFirst(tokens))).orElse(Boolean.FALSE)) {
                throw new IllegalArgumentException("Illegal preposition");
            }
            deleteFirst(tokens);
        }
        String listPointer = lemmatize(getFirst(tokens));
        if (!"список".equals(listPointer)) {
            throw new IllegalArgumentException("Illegal list pointer");
        }
        deleteFirst(tokens);
        if (tokens.isEmpty()) {
            throw new IllegalArgumentException("List name is not defined");
        }
        String listName = getFirst(tokens);
        deleteFirst(tokens);
        List<String> itemNameParts = intent != UserRequest.Intent.SEND_LIST ? tokens : lemmatize(tokens);
        List<String> items = new ArrayList<>();
        if (!itemNameParts.isEmpty()) {
            items.add(itemNameParts.stream().collect(Collectors.joining(" ")));
        }
        return new UserRequest(intent, Optional.of(listName), items);
    }

    private List<String> lemmatize(List<String> words) {
        return words.stream().map(this::lemmatize).collect(Collectors.toList());
    }

    private String lemmatize(String word) {
        return Lemmer.analyzeWord(word.toCharArray())[0];
    }

    private void deleteFirst(List<String> words) {
        words.remove(0);
    }

    private String getFirst(List<String> words) {
        return words.get(0);
    }

}
