package ru.yandex.mail.hackathon;

import java.net.URI;
import java.util.Optional;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;

import ru.yandex.mail.hackathon.calendar.CreateTodoItemResponse;
import ru.yandex.mail.hackathon.calendar.GetTodoListResponse;
import ru.yandex.mail.hackathon.calendar.GetTodoListsResponse;
import ru.yandex.mail.hackathon.calendar.TodoList;


/**
 * Here we store all the operations interacting with calendar api.
 */
public class CalendarFacadeImpl implements CalendarFacade {
    private final String url;
    private final HttpClient client;
    private final ObjectMapper mapper;

    public CalendarFacadeImpl(final String url, final HttpClient client) {
        this.url = url;
        this.client = client;
        mapper = new ObjectMapper();
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    public GetTodoListsResponse getTodoLists(long uid) {
        try {
            URI uri = new URIBuilder(url)
                    .setPath("/get-todo-lists.json")
                    .addParameter("uid", String.valueOf(uid))
                    .build();

            HttpGet request = new HttpGet(uri);

            return client.execute(request, response -> mapper.readValue(response.getEntity().getContent(), GetTodoListsResponse.class));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Optional<String> getTodoList(long uid, String title) {
        try {
            GetTodoListsResponse lists = getTodoLists(uid);
            return lists.getItems().stream().filter(list -> title.equals(list.getTitle())).findFirst().map(TodoList::getExternalId);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String createTodoList(long uid, String title) {
        try {
            URI uri = new URIBuilder(url)
                    .setPath("/create-todo-list.json")
                    .addParameter("uid", String.valueOf(uid))
                    .addParameter("title", title)
                    .build();

            CreateTodoListResponse result = client.execute(new HttpPost(uri), response -> mapper.readValue(response.getEntity().getContent(), CreateTodoListResponse.class));

            uri = new URIBuilder(url)
                    .setPath("/get-todo-list.json")
                    .addParameter("uid", String.valueOf(uid))
                    .addParameter("list-id", String.valueOf(result.getId()))
                    .build();

            return client.execute(new HttpGet(uri), response -> mapper.readValue(response.getEntity().getContent(), GetTodoListResponse.class).getTodoList().getExternalId());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public long createTodoItem(long uid, String externalId, String title) {
        try {
            URI uri = new URIBuilder(url)
                    .setPath("/create-todo-item.json")
                    .addParameter("uid", String.valueOf(uid))
                    .addParameter("title", title)
                    .addParameter("list-external-id", externalId)
                    .build();
            return client.execute(new HttpPost(uri), response -> mapper.readValue(response.getEntity().getContent(), CreateTodoItemResponse.class).getTodoItem().getId());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
