package ru.yandex.mail.hackathon;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Arrays;

import com.google.protobuf.Any;
import com.google.protobuf.CodedOutputStream;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.alice.megamind.protos.scenarios.RequestProto.TScenarioRunRequest;
import ru.yandex.alice.megamind.protos.scenarios.ResponseProto.TLayout;
import ru.yandex.alice.megamind.protos.scenarios.ResponseProto.TScenarioResponseBody;
import ru.yandex.alice.megamind.protos.scenarios.ResponseProto.TScenarioRunResponse;
import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.client.tvm2.UserAuthResult;
import ru.yandex.collection.Pattern;
import ru.yandex.http.config.ClientHttpsConfigBuilder;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.ForbiddenException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.mail.hackathon.config.ImmutableHackathonConfig;
import ru.yandex.mail.hackathon.todolistmodifier.TodoListModifier;
import ru.yandex.mail.hackathon.todolistmodifier.TodoListModifyState;

public class Hackathon
    extends HttpProxy<ImmutableHackathonConfig>
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final ContentType APPLICATION_PROTOBUF =
        ContentType.create("application/protobuf");

    private final CalendarFacadeImpl calendarClient;
    private final BlackboxClient blackboxClient;
    private final AsyncClient msearchProxyClient;

    public Hackathon(final ImmutableHackathonConfig config)
        throws IOException
    {
        super(config);
        HttpTargetConfigBuilder targetConfig = new HttpTargetConfigBuilder();
        targetConfig.connections(10);
        ClientHttpsConfigBuilder httpsConfig = new ClientHttpsConfigBuilder();
        httpsConfig.verifyHostname(false);
        httpsConfig.verifyCertificate(false);
        targetConfig.httpsConfig(httpsConfig);
        try {
            calendarClient =
                new CalendarFacadeImpl(
                    config.calendarHost(),
                    ClientBuilder.createClient(targetConfig.build()));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        blackboxClient =
            registerClient(
                "Blackbox",
                new BlackboxClient(reactor, config.blackboxConfig()),
                config.blackboxConfig());
        msearchProxyClient =
            client("MSearchProxy", config.msearchProxyConfig());
        register(new Pattern<>("", true), this);
    }

    public CalendarFacadeImpl calendarClient() {
        return calendarClient;
    }

    public BlackboxClient blackboxClient() {
        return blackboxClient;
    }

    public AsyncClient msearchProxyClient() {
        return msearchProxyClient;
    }

    @Override
    public BasicAsyncRequestConsumer processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new BasicAsyncRequestConsumer();
    }

    private long extractUid(final HttpRequest request) throws HttpException {
        if (serviceContextRenewalTask == null) {
            throw new ForbiddenException("No service context configured");
        }
        UserAuthResult authResult =
            serviceContextRenewalTask.checkUserAuthorization(
                request,
                YandexHeaders.X_YA_USER_TICKET);
        String errorDescription = authResult.errorDescription();
        if (errorDescription != null) {
            throw new ForbiddenException(errorDescription);
        }

        long[] uids = authResult.ticket().getUids();
        if (uids.length != 1) {
            throw new ForbiddenException(
                "One uid expected in ticket, got following uids: "
                + Arrays.toString(uids));
        }
        return uids[0];
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        long uid = extractUid(request);
        ProxySession session = new BasicProxySession(this, exchange, context);
        if (request instanceof HttpEntityEnclosingRequest) {
            DecodableByteArrayOutputStream in =
                new DecodableByteArrayOutputStream();
            ((HttpEntityEnclosingRequest) request).getEntity().writeTo(in);
            TScenarioRunRequest runRequest =
                TScenarioRunRequest.parseFrom(in.toByteArray());
            String text =
                runRequest.getInput().getText().getUtterance();
            if (text.isEmpty()) {
                text =
                    runRequest.getInput().getVoice().getUtterance();
            }
            session.logger().info("Input text is <" + text + '>');
            session.logger().info(
                "Has state: " + runRequest.getBaseRequest().hasState());
            String state;
            if (runRequest.getBaseRequest().hasState()) {
                String type =
                    runRequest.getBaseRequest().getState().getTypeUrl();
                session.logger().info("State has type url: " + type);
                if (type.isEmpty()) {
                    state = "";
                } else {
                    state =
                        runRequest.getBaseRequest()
                            .getState().unpack(State.class).getText();
                }
            } else {
                state = "";
            }

            session.logger().info("State = " + state);
            TodoListModifier modifier =
                new TodoListModifier(session.logger(), calendarClient);
            TodoListModifyState modifyState =
                modifier.process(uid, text, state);

            TLayout.Builder layoutBuilder = TLayout.newBuilder();
            layoutBuilder.addCards(
                TLayout.TCard.newBuilder()
                    .setText(modifyState.getResponseToUser())
                    .build());
            layoutBuilder.setOutputSpeech(modifyState.getResponseToUser());

            String responseState = modifyState.toJsonString();
            session.logger().info("Response string: " + modifyState.getResponseToUser());
            session.logger().info("Response state: " + responseState);

            TScenarioResponseBody.Builder responseBodyBuilder =
                TScenarioResponseBody.newBuilder();
            responseBodyBuilder.setLayout(layoutBuilder.build());
            if (responseState != null) {
                responseBodyBuilder.setState(
                    Any.pack(
                        State.newBuilder().setText(responseState).build()));
            }

            TScenarioRunResponse.Builder responseBuilder =
                TScenarioRunResponse.newBuilder();
            responseBuilder.setResponseBody(responseBodyBuilder.build());

            ByteArrayOutputStream out = new ByteArrayOutputStream();
            CodedOutputStream coded = CodedOutputStream.newInstance(out);
            responseBuilder.build().writeTo(coded);
            coded.flush();
            session.response(
                HttpStatus.SC_OK,
                new NByteArrayEntity(
                    out.toByteArray(),
                    APPLICATION_PROTOBUF));
        } else {
            session.response(HttpStatus.SC_METHOD_NOT_ALLOWED);
        }
    }
}

