package ru.yandex.parser.html;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Deque;
import java.util.Locale;
import java.util.Map;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

public abstract class HtmlProcessorBase {
    private static final int INITIAL_ENTBUFFER_CAPACITY = 32;
    private static final int INITIAL_BUFFER_CAPACITY = 1024;
    private static final int HEX = 16;
    private static final int READ_BUFFER_SIZE = 8192;
    private static final String URI = "http://www.w3.org/1999/xhtml";
    private static final String HTML = "html";
    private static final String BODY = "body";
    private static final char[] IGNORABLE_WHITESPACE = new char[] {'\n'};
    private static final Map<String, String> ENTITIES =
        HtmlEntities.DEFAULT.entities();

    private final AttributesImpl attrs = new AttributesImpl();
    private final Deque<String> tagsStack = new ArrayDeque<String>();
    private final Attributes noAttrs = new AttributesImpl();
    private final ContentHandler handler;
    private char[] entbuffer = new char[INITIAL_ENTBUFFER_CAPACITY];
    private int entbuflen = 0;
    private char[] buffer = new char[INITIAL_BUFFER_CAPACITY];
    private int buflen = 0;
    private String tag = null;
    private String attrName = null;

    protected HtmlProcessorBase(final ContentHandler handler) {
        this.handler = handler;
    }

    protected void clearBuffer() {
        buflen = 0;
    }

    protected void clearEntityBuffer() {
        entbuflen = 0;
    }

    protected void clearAttributes() {
        attrs.clear();
    }

    protected void addCharEntity(final char c) {
        addCharBuf(entbuffer, entbuflen, c);
    }

    private void ensureCapacity(final int capacity) {
        if (capacity >= buffer.length) {
            buffer = Arrays.copyOf(
                buffer,
                Math.max(capacity, buffer.length << 1));
        }
    }

    protected void addChar(final char c) {
        ensureCapacity(buflen + 1);
        buffer[buflen++] = c;
    }

    private void addCharBuf(final char[] cbuf, final int len, final char c) {
        ensureCapacity(buflen + len + 1);
        for (int i = 0; i < len;) {
            buffer[buflen++] = cbuf[i++];
        }
        buffer[buflen++] = c;
    }

    private void addString(final String str) {
        ensureCapacity(buflen + str.length());
        for (int i = 0; i < str.length(); ++i) {
            buffer[buflen++] = str.charAt(i);
        }
    }

    private void checkBody() throws SAXException {
        if (tagsStack.isEmpty()) {
            handler.startElement(URI, HTML, HTML, noAttrs);
            tagsStack.addLast(HTML);
            handler.startElement(URI, BODY, BODY, noAttrs);
            tagsStack.addLast(BODY);
        }
    }

    private void flushBuffer() throws SAXException {
        checkBody();
        handler.characters(buffer, 0, buflen);
    }

    protected void addCharNoResize(final char c) throws SAXException {
        if (buflen >= buffer.length) {
            flushBuffer();
            buffer[0] = c;
            buflen = 1;
        } else {
            buffer[buflen++] = c;
        }
    }

    protected void addCharNoResize(final char c1, final char c2)
        throws SAXException
    {
        if (buflen + 1 >= buffer.length) {
            flushBuffer();
            buffer[0] = c1;
            buffer[1] = c2;
            buflen = 2;
        } else {
            buffer[buflen++] = c1;
            buffer[buflen++] = c2;
        }
    }

    protected void addCharNoResize(final char c1, final char c2, final char c3)
        throws SAXException
    {
        if (buflen + 2 >= buffer.length) {
            flushBuffer();
            buffer[0] = c1;
            buffer[1] = c2;
            buffer[2] = c3;
            buflen = 2 + 1;
        } else {
            buffer[buflen++] = c1;
            buffer[buflen++] = c2;
            buffer[buflen++] = c3;
        }
    }

    private boolean addCodePoint(final int codePoint) {
        if (Character.isSupplementaryCodePoint(codePoint)) {
            ensureCapacity(buflen + 2);
            buffer[buflen++] = Character.highSurrogate(codePoint);
            buffer[buflen++] = Character.lowSurrogate(codePoint);
        } else if (Character.isValidCodePoint(codePoint)
            && !Character.isSurrogate((char) codePoint))
        {
            addChar((char) codePoint);
        } else {
            return false;
        }
        return true;
    }

    protected void addEntChar(final char c) {
        if (entbuflen >= entbuffer.length) {
            entbuffer = Arrays.copyOf(buffer, entbuflen << 1);
        }
        entbuffer[entbuflen++] = c;
    }

    private void ignorableWhitespace() throws SAXException {
        handler.ignorableWhitespace(IGNORABLE_WHITESPACE, 0, 1);
    }

    protected void barrier() throws SAXException {
        int spaces = 0;
        for (int i = 0; i < buflen; ++i) {
            if (buffer[i] <= ' ') {
                ++spaces;
            }
        }
        if (buflen > spaces) {
            flushBuffer();
        }
        buflen = 0;
    }

    protected void storeName() {
        tag = new String(buffer, 0, buflen).toLowerCase(Locale.ENGLISH);
        buflen = 0;
    }

    protected void storeAttrName() {
        attrName = new String(buffer, 0, buflen).toLowerCase(Locale.ENGLISH);
        buflen = 0;
    }

    protected void storeTagValue() {
        attrs.addAttribute(
            "",
            attrName,
            attrName,
            "CDATA",
            new String(buffer, 0, buflen));
        buflen = 0;
    }

    protected void emptyTag() throws SAXException {
        handler.startElement(URI, tag, tag, attrs);
        handler.endElement(URI, tag, tag);
    }

    protected void tagStart() throws SAXException {
        ignorableWhitespace();
        if (tag.equals(HTML)) {
            String prev;
            while ((prev = tagsStack.pollLast()) != null) {
                handler.endElement(URI, prev, prev);
            }
        } else {
            checkBody();
        }
        handler.startElement(URI, tag, tag, attrs);
        tagsStack.addLast(tag);
    }

    protected void tagEnd() throws SAXException {
        if (tagsStack.contains(tag)) {
            while (true) {
                String prev = tagsStack.pollLast();
                if (prev == null || prev.equals(tag)) {
                    break;
                }
                handler.endElement(URI, prev, prev);
            }
            handler.endElement(URI, tag, tag);
        }
        ignorableWhitespace();
    }

    protected void documentStart() throws SAXException {
        handler.startDocument();
        handler.startPrefixMapping("", URI);
    }

    protected void documentEnd() throws SAXException {
        String prev;
        while ((prev = tagsStack.pollLast()) != null) {
            handler.endElement(URI, prev, prev);
        }
        handler.endDocument();
    }

    protected void charRef(final char c) {
        boolean done = false;
        if (entbuflen > 2) {
            if (entbuffer[1] == '#') {
                try {
                    if (entbuffer[2] == 'x' || entbuffer[2] == 'X') {
                        if (entbuflen > 2 + 1) {
                            done =
                                addCodePoint(
                                    Integer.parseInt(
                                        new String(
                                            entbuffer,
                                            2 + 1,
                                            entbuflen - 2 - 1),
                                        HEX));
                        }
                    } else {
                        done = addCodePoint(
                            Integer.parseInt(
                                new String(entbuffer, 2, entbuflen - 2)));
                    }
                } catch (NumberFormatException e) {
                    done = false;
                }
            } else {
                String str = ENTITIES.get(
                    new String(entbuffer, 1, entbuflen - 1));
                if (str != null) {
                    addString(str);
                    done = true;
                }
            }
        }
        if (!done) {
            addCharBuf(entbuffer, entbuflen, c);
        }
    }

    public void process(final Reader reader) throws IOException, SAXException {
        char[] buffer = new char[READ_BUFFER_SIZE];
        int read;
        while ((read = reader.read(buffer)) != -1) {
            process(buffer, 0, read);
        }
        process(null, 0, 0);
    }

    public void process(final char[] data) throws SAXException {
        process(data, 0, data.length);
    }

    public abstract void process(
        final char[] data,
        final int off,
        final int len)
        throws SAXException;
}

