package ru.yandex.sanitizer2;

import java.util.Collection;
import java.util.List;
import java.util.regex.Matcher;

import com.google.common.base.Function;
import com.helger.css.decl.CSSDeclaration;
import com.helger.css.decl.IHasCSSDeclarations;
import org.owasp.html.CssGrammar;
import org.owasp.html.CssSchema;
import org.owasp.html.CssTokens;
import org.owasp.html.StylingPolicy;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.collection.EqualableIdentityHashMap;
import ru.yandex.sanitizer2.config.ImmutablePropertyAliasConfig;
import ru.yandex.sanitizer2.config.ImmutablePropertyConfig;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.sanitizer2.config.ImmutableShorthandPropertyConfig;

public class BasicStyle
    extends EqualableIdentityHashMap<String, CssDeclaration>
    implements Style
{
    private static final long serialVersionUID = 0L;
    private static final Function<String, String> IDENTITY_FUNCTION = x -> x;

    public BasicStyle() {
    }

    public BasicStyle(final Style style) {
        super(style);
    }

    public BasicStyle(final Collection<CssDeclaration> declarations) {
        super(declarations.size() << 1);
        merge(declarations);
    }

    // CSOFF: ParameterNumber
    public static void addProperty(
        final ImmutableSanitizingConfig config,
        final String property,
        final String expression,
        final boolean isImportant,
        final List<CssDeclaration> declarations,
        final SanitizingContext context)
    {
        ImmutablePropertyConfig propertyConfig = config.style().get(property);
        if (propertyConfig != null) {
            String value = propertyConfig.sanitize(expression, context);
            if (value != null) {
                UrlRewriter urlRewriter = propertyConfig.urlRewriter();
                Function<String, String> urlProcessor;
                if (urlRewriter == null) {
                    urlProcessor = IDENTITY_FUNCTION;
                } else {
                    urlProcessor = url -> urlRewriter.sanitize(url, context);
                }
                StringBuilder sb = context.temporaryStringBuilder();
                CssUrlExtractor urlExtractor = new CssUrlExtractor(
                    sb,
                    urlProcessor,
                    context.temporaryUrlsList(),
                    config);
                urlExtractor.startProperty(property);
                sb.setLength(0);
                CssTokens lex =
                    CssTokens.lex(value, context.temporaryTokenTypes());
                CssGrammar.parsePropertyValue(lex.iterator(), urlExtractor);
                urlExtractor.endProperty();
                if (sb.length() > 0) {
                    declarations.add(
                        new CssDeclaration(
                            property,
                            sb.toString(),
                            isImportant,
                            propertyConfig,
                            CollectionCompactor.compactOrCopy(
                                urlExtractor.urls)));
                }
            }
        }
    }

    public static void processDeclaration(
        final ImmutableSanitizingConfig config,
        final String property,
        final String expression,
        final boolean isImportant,
        final List<CssDeclaration> declarations,
        final SanitizingContext context)
    {
        ImmutableShorthandPropertyConfig shorthandConfig =
            config.shorthands().get(property);
        if (shorthandConfig == null) {
            addProperty(
                config,
                property,
                expression,
                isImportant,
                declarations,
                context);
        } else {
            Matcher matcher = shorthandConfig.pattern().matcher(expression);
            if (matcher.matches()) {
                List<ImmutableShorthandPropertyConfig.Shorthand> shorthands =
                    shorthandConfig.propertiesList();
                for (int i = 0; i < shorthands.size(); ++i) {
                    ImmutableShorthandPropertyConfig.Shorthand shorthand =
                        shorthands.get(i);
                    String group = matcher.group(shorthand.property());
                    ImmutablePropertyAliasConfig aliasConfig =
                        shorthand.aliasConfig();
                    if (group == null) {
                        group = aliasConfig.defaultValue();
                    }
                    if (group != null) {
                        List<String> names = aliasConfig.namesList();
                        for (int j = 0; j < names.size(); ++j) {
                            String alias = names.get(j);
                            ImmutableShorthandPropertyConfig nestedConfig;
                            if (alias.equals(property)) {
                                nestedConfig = null;
                            } else {
                                nestedConfig = config.shorthands().get(alias);
                            }
                            if (nestedConfig == null) {
                                addProperty(
                                    config,
                                    alias,
                                    group,
                                    isImportant,
                                    declarations,
                                    context);
                            } else {
                                processDeclaration(
                                    config,
                                    alias,
                                    group,
                                    isImportant,
                                    declarations,
                                    context);
                            }
                        }
                        if (aliasConfig.terminal()) {
                            break;
                        }
                    }
                }
            }
        }
    }
    // CSON: ParameterNumber

    public static void convertDeclarations(
        final List<CssDeclaration> declarations,
        final ImmutableSanitizingConfig config,
        final IHasCSSDeclarations<?> declarationList,
        final SanitizingContext context)
    {
        int declarationsCount = declarationList.getDeclarationCount();
        for (int i = 0; i < declarationsCount; ++i) {
            CSSDeclaration declaration =
                declarationList.getDeclarationAtIndex(i);
            processDeclaration(
                config,
                config.internProperty(declaration.getProperty()),
                declaration.getExpressionAsCSSString(),
                declaration.isImportant(),
                declarations,
                context);
        }
    }

    public void merge(final Style style) {
        merge(style.values());
    }

    public void merge(final Collection<CssDeclaration> declarations) {
        for (CssDeclaration declaration: declarations) {
            String property = declaration.property();
            if (declaration.important()) {
                put(property, declaration);
            } else {
                CssDeclaration old = get(property);
                if (old == null || !old.important()) {
                    put(property, declaration);
                }
            }
        }
    }

    public static Style merge(final Style lhs, final Style rhs) {
        Style result;
        if (lhs.isEmpty()) {
            result = rhs;
        } else if (rhs.isEmpty()) {
            result = lhs;
        } else {
            BasicStyle style = new BasicStyle(lhs);
            style.merge(rhs);
            result = style.compact();
        }
        return result;
    }

    public Style compact() {
        Style style;
        switch (size()) {
            case 0:
                style = EmptyStyle.INSTANCE;
                break;
            case 1:
                style = new SingletonStyle(values().iterator().next());
                break;
            default:
                style = this;
                break;
        }
        return style;
    }

    @Override
    public boolean allInherited() {
        for (CssDeclaration declaration: values()) {
            if (!declaration.config().inherited()) {
                return false;
            }
        }
        return true;
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        boolean empty = true;
        for (CssDeclaration declaration: values()) {
            if (empty) {
                empty = false;
            } else {
                sb.append(';');
            }
            declaration.toStringBuilder(sb);
        }
    }

    private static class CssUrlExtractor
        extends StylingPolicy.CssSanitizer
    {
        private final List<AttrUrlInfo> urls;
        private final ImmutableSanitizingConfig config;

        CssUrlExtractor(
            final StringBuilder sb,
            final Function<String, String> urlRewriter,
            final List<AttrUrlInfo> urls,
            final ImmutableSanitizingConfig config)
        {
            super(sb, CssSchema.DEFAULT, urlRewriter);
            this.urls = urls;
            this.config = config;
        }

        @Override
        protected void appendSanitizedUrl(final String url) {
            int start = sanitizedCss.length();
            super.appendSanitizedUrl(url);
            urls.add(
                UrlType.AUTO.detectUrlClass(
                    start,
                    sanitizedCss.length(),
                    url,
                    config));
        }
    }
}

