package ru.yandex.sanitizer2;

import java.nio.charset.CharacterCodingException;

import ru.yandex.charset.Encoder;
import ru.yandex.function.StringBuilderVoidProcessor;
import ru.yandex.io.DecodableByteArrayOutputStream;

public class ByteArrayHtmlCollector
    implements HtmlCollector<CharacterCodingException>
{
    private static final int FLUSH_INTERVAL = 4096;

    // Encodes chars to bytes
    private final Encoder encoder;
    private final StringBuilderVoidProcessor<byte[], CharacterCodingException>
        sbEncoder;
    // Accumulates data between flushes
    private final StringBuilder sb;
    // Accumulates flushed data
    private final DecodableByteArrayOutputStream data;

    public ByteArrayHtmlCollector(
        final Encoder encoder,
        final int inputLength)
    {
        this.encoder = encoder;
        sbEncoder = new StringBuilderVoidProcessor<>(encoder);
        sb = new StringBuilder(Math.min(inputLength, FLUSH_INTERVAL) << 1);
        data = new DecodableByteArrayOutputStream(inputLength << 1);
    }

    public DecodableByteArrayOutputStream data() {
        return data;
    }

    @Override
    public int length() {
        return data.length();
    }

    @Override
    public StringBuilder sb() {
        return sb;
    }

    private void forceFlush() throws CharacterCodingException {
        sbEncoder.process(sb);
        sbEncoder.processWith(data);
        encoder.ensureMaxBufCapacity(FLUSH_INTERVAL);
        sb.setLength(0);
    }

    @Override
    public void flush() throws CharacterCodingException {
        if (sb.length() > 0) {
            forceFlush();
        }
    }

    @Override
    public void append(final char c) {
        sb.append(c);
    }

    @Override
    public void append(final String str) {
        sb.append(str);
    }

    @Override
    public void append(final StringBuilder sb) {
        this.sb.append(sb);
    }

    @Override
    public void appendAndFlush(final char c) throws CharacterCodingException {
        sb.append(c);
        if (sb.length() > FLUSH_INTERVAL) {
            forceFlush();
        }
    }

    @Override
    public void appendAndFlush(final String str)
        throws CharacterCodingException
    {
        sb.append(str);
        if (sb.length() > FLUSH_INTERVAL) {
            forceFlush();
        }
    }
}

