package ru.yandex.sanitizer2;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.CharacterCodingException;
import java.util.Set;
import java.util.function.Predicate;

import ru.yandex.net.uri.fast.FastUri;
import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;
import ru.yandex.sanitizer2.config.ImmutableHiderefererConfig;
import ru.yandex.util.string.StringUtils;

public class HiderefererUrlRewriter extends AbstractUrlRewriter {
    private static final int DEFAULT_MAX_CAPACITY = 1024;
    private static final ThreadLocal<PctEncoder.CompactEncoder> ENCODER =
        ThreadLocal.withInitial(
            () -> new PctEncoder.CompactEncoder(
                DEFAULT_MAX_CAPACITY,
                PctEncodingRule.QUERY));

    private final Predicate<String> whitelistHosts;
    private final Predicate<String> blacklistHosts;
    private final String scheme;
    private final String userInfo;
    private final String host;
    private final int port;
    private final String path;
    private final String query;
    private final String fragment;
    private final String schemeSpecificPart;
    private final String authority;

    public HiderefererUrlRewriter(
        final Set<String> bypassSchemes,
        final Set<String> allowedSchemes,
        final UrlRewriter next,
        final ImmutableHiderefererConfig config)
    {
        super(bypassSchemes, allowedSchemes, next);
        whitelistHosts = config.whitelistHosts();
        blacklistHosts = config.blacklistHosts();
        URI uri = config.hidereferer();
        scheme = uri.getScheme();
        userInfo = uri.getRawUserInfo();
        host = uri.getHost();
        port = uri.getPort();
        path = uri.getRawPath();
        String query = uri.getRawQuery();
        if (query == null) {
            this.query = "";
        } else {
            this.query = StringUtils.concat(query, '&');
        }
        fragment = uri.getRawFragment();
        schemeSpecificPart = uri.getRawSchemeSpecificPart();
        authority = uri.getRawAuthority();
    }

    @Override
    public FastUri sanitize(final FastUri uri, final SanitizingContext context)
        throws URISyntaxException
    {
        FastUri tmp;
        String host = uri.host();
        if (host != null
            && (!whitelistHosts.test(host) || blacklistHosts.test(host)))
        {
            StringBuilder uriSb =
                new StringBuilder(uri.expectedStringLength());
            uri.toStringBuilder(uriSb);
            StringBuilder sb = new StringBuilder(query);
            try {
                ENCODER.get().process(uriSb, sb);
            } catch (CharacterCodingException e) {
                return null;
            }
            tmp = new FastUri(
                scheme,
                userInfo,
                this.host,
                port,
                path,
                new String(sb),
                fragment,
                schemeSpecificPart,
                authority);
        } else {
            tmp = uri;
        }
        return next.sanitize(tmp, context);
    }
}

