package ru.yandex.sanitizer2;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.owasp.html.HtmlTextEscapingMode;

import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.sanitizer2.config.ImmutableTagConfig;

public class HtmlCompactor
    implements HtmlNodeVisitor<HtmlNode, RuntimeException>
{
    private static final List<String> EMPTY_CLASSES = Collections.emptyList();

    private final Map<String, ImmutableTagConfig> tags;

    public HtmlCompactor(final ImmutableSanitizingConfig config) {
        tags = config.tags();
    }

    private static void addNode(
        final List<HtmlNode> nodes,
        final HtmlNode node)
    {
        if (node instanceof HtmlTag) {
            HtmlTag tag = (HtmlTag) node;
            if (tag.tagName().isEmpty()) {
                int size = tag.size();
                for (int i = 0; i < size; ++i) {
                    addNode(nodes, tag.get(i));
                }
            } else {
                nodes.add(tag);
            }
        } else {
            nodes.add(node);
        }
    }

    private static void addNode(
        final HtmlTag parentTag,
        final HtmlNode node)
    {
        if (node instanceof HtmlTag) {
            HtmlTag tag = (HtmlTag) node;
            if (tag.tagName().isEmpty()) {
                int size = tag.size();
                for (int i = 0; i < size; ++i) {
                    addNode(parentTag, tag.get(i));
                }
            } else {
                parentTag.addNode(tag);
            }
        } else {
            parentTag.addNode(node);
        }
    }

    @Override
    public HtmlTag visit(final HtmlTag tag) {
        String tagName = tag.tagName();
        int size = tag.size();
        if (size == 0) {
            HtmlTag result;
            if (tagName.isEmpty()) {
                result = HtmlTag.EMPTY_TAG;
            } else {
                ImmutableTagConfig config = tags.get(tagName);
                if (config.requireContent()) {
                    result = HtmlTag.EMPTY_TAG;
                } else if (config.requireAttrs()
                    && tag.id() == null
                    && tag.classes().isEmpty()
                    && tag.attrs().isEmpty()
                    && tag.style().isEmpty())
                {
                    result = HtmlTag.EMPTY_TAG;
                } else {
                    result = tag;
                }
            }
            return result;
        }
        List<HtmlNode> nodes = new ArrayList<>(size);
        for (int i = 0; i < size; ++i) {
            addNode(nodes, tag.get(i).accept(this));
        }
        if (tagName.isEmpty()) {
            return new HtmlTag(
                nodes,
                tag.escaping(),
                "",
                tag.classes(),
                tag.sanitizedClasses(),
                tag.id(),
                tag.sanitizedId(),
                tag.attrs(),
                tag.style());
        }
        ImmutableTagConfig config = tags.get(tagName);
        HtmlTag result = null;
        if (!nodes.isEmpty() && tag.style().allInherited()) {
            if (config.flag()) {
                if (tag.id() == null
                    && tag.attrs().isEmpty()
                    && tag.classes().isEmpty())
                {
                    result = tryCollapseFlag(tag, nodes);
                }
            } else if (config.combine()
                && tag.id() == null
                && tag.classes().isEmpty())
            {
                if (tag.attrs().isEmpty()) {
                    result = tryCombineStyle(tag, nodes);
                } else if (tag.style().isEmpty()) {
                    result = tryCombineAttrs(tag, nodes);
                }
            }
        }
        if (result == null) {
            result = new HtmlTag(
                nodes,
                tag.escaping(),
                tagName,
                tag.classes(),
                tag.sanitizedClasses(),
                tag.id(),
                tag.sanitizedId(),
                tag.attrs(),
                tag.style());
        }
        if (!result.isEmpty()) {
            tryCompactNodes(result);
        }
        if (!result.tagName().isEmpty()) {
            config = tags.get(result.tagName());
            if (config.requireContent() && result.isEmpty()) {
                result = HtmlTag.EMPTY_TAG;
            } else if (config.requireAttrs()
                && result.id() == null
                && result.classes().isEmpty()
                && result.attrs().isEmpty()
                && result.style().isEmpty())
            {
                result = new HtmlTag(
                    result,
                    HtmlTextEscapingMode.PCDATA,
                    "",
                    EMPTY_CLASSES,
                    EMPTY_CLASSES,
                    null,
                    null,
                    EmptyAttrs.INSTANCE,
                    EmptyStyle.INSTANCE);
            }
        }
        return result;
    }

    @Override
    public HtmlCDataTag visit(final HtmlCDataTag cdata) {
        return cdata;
    }

    @Override
    public HtmlText visit(final HtmlText text) {
        return text;
    }

    // Implies that:
    // tag.id() == null && tag.classes().isEmpty() && tag.attrs().isEmpty()
    private HtmlTag tryCollapseFlag(
        final HtmlTag tag,
        final List<HtmlNode> nodes)
    {
        HtmlTag result = null;
        boolean collapsable = false;
        int size = nodes.size();
        for (int i = 0; i < size; ++i) {
            HtmlNode node = nodes.get(i);
            if (node instanceof HtmlTag) {
                HtmlTag subtag = (HtmlTag) node;
                if (subtag.id() == null
                    && subtag.classes().isEmpty()
                    && subtag.attrs().isEmpty()
                    && subtag.style().allInherited()
                    && tags.get(subtag.tagName()).flag()
                    && tag.tagName().compareTo(subtag.tagName()) >= 0)
                {
                    collapsable = true;
                    break;
                }
            }
        }
        if (collapsable) {
            result = new HtmlTag(
                HtmlTextEscapingMode.PCDATA,
                "",
                EMPTY_CLASSES,
                EMPTY_CLASSES,
                null,
                null,
                EmptyAttrs.INSTANCE,
                EmptyStyle.INSTANCE);
            for (int i = 0; i < size; ++i) {
                HtmlNode node = nodes.get(i);
                boolean processed = false;
                if (node instanceof HtmlTag) {
                    HtmlTag subtag = (HtmlTag) node;
                    if (subtag.id() == null
                        && subtag.classes().isEmpty()
                        && subtag.attrs().isEmpty()
                        && subtag.style().allInherited()
                        && tags.get(subtag.tagName()).flag())
                    {
                        int cmp =
                            tag.tagName().compareTo(subtag.tagName());
                        if (cmp == 0) {
                            // Collapse tags
                            HtmlTag newTag = new HtmlTag(
                                subtag,
                                tag.escaping(),
                                tag.tagName(),
                                EMPTY_CLASSES,
                                EMPTY_CLASSES,
                                null,
                                null,
                                EmptyAttrs.INSTANCE,
                                BasicStyle.merge(tag.style(), subtag.style()));
                            result.addNode(newTag);
                            processed = true;
                        } else if (cmp > 0) {
                            // Reorder tags and compact once again,
                            // so duplicate flags can be collapsed
                            HtmlTag lowerTag = new HtmlTag(
                                subtag,
                                tag.escaping(),
                                tag.tagName(),
                                EMPTY_CLASSES,
                                EMPTY_CLASSES,
                                null,
                                null,
                                EmptyAttrs.INSTANCE,
                                tag.style());
                            HtmlTag upperTag = new HtmlTag(
                                1,
                                subtag.escaping(),
                                subtag.tagName(),
                                EMPTY_CLASSES,
                                EMPTY_CLASSES,
                                null,
                                null,
                                EmptyAttrs.INSTANCE,
                                subtag.style());
                            upperTag.add(lowerTag);
                            // Try reorder and collapse flags once again
                            addNode(result, upperTag.accept(this));
                            processed = true;
                        }
                    }
                }
                if (!processed) {
                    HtmlTag newTag = new HtmlTag(
                        tag.escaping(),
                        tag.tagName(),
                        EMPTY_CLASSES,
                        EMPTY_CLASSES,
                        null,
                        null,
                        tag.attrs(),
                        tag.style());
                    newTag.addNode(node);
                    result.addNode(newTag);
                }
            }
        }
        return result;
    }

    // Implies that:
    // tag.id() == null && tag.classes().isEmpty() && tag.attrs().isEmpty()
    private HtmlTag tryCombineStyle(
        final HtmlTag tag,
        final List<HtmlNode> nodes)
    {
        HtmlTag result = null;
        boolean combinable = false;
        int size = nodes.size();
        for (int i = 0; i < size; ++i) {
            HtmlNode node = nodes.get(i);
            if (node instanceof HtmlTag) {
                HtmlTag subtag = (HtmlTag) node;
                if (subtag.id() == null
                    && subtag.classes().isEmpty()
                    && subtag.attrs().isEmpty()
                    && tag.tagName().equals(subtag.tagName())
                    && subtag.style().allInherited())
                {
                    combinable = true;
                    break;
                }
            }
        }
        if (combinable) {
            result = new HtmlTag(
                HtmlTextEscapingMode.PCDATA,
                "",
                EMPTY_CLASSES,
                EMPTY_CLASSES,
                null,
                null,
                EmptyAttrs.INSTANCE,
                EmptyStyle.INSTANCE);
            for (int i = 0; i < size; ++i) {
                HtmlNode node = nodes.get(i);
                boolean processed = false;
                if (node instanceof HtmlTag) {
                    HtmlTag subtag = (HtmlTag) node;
                    if (subtag.id() == null
                        && subtag.classes().isEmpty()
                        && subtag.attrs().isEmpty()
                        && tag.tagName().equals(subtag.tagName())
                        && subtag.style().allInherited())
                    {
                        HtmlTag newTag = new HtmlTag(
                            subtag,
                            tag.escaping(),
                            tag.tagName(),
                            EMPTY_CLASSES,
                            EMPTY_CLASSES,
                            null,
                            null,
                            tag.attrs(),
                            BasicStyle.merge(tag.style(), subtag.style()));
                        result.addNode(newTag);
                        processed = true;
                    }
                }
                if (!processed) {
                    HtmlTag newTag = new HtmlTag(
                        tag.escaping(),
                        tag.tagName(),
                        EMPTY_CLASSES,
                        EMPTY_CLASSES,
                        null,
                        null,
                        tag.attrs(),
                        tag.style());
                    newTag.addNode(node);
                    result.addNode(newTag);
                }
            }
        }
        return result;
    }

    // Implies that:
    // tag.id() == null && tag.classes().isEmpty() && tag.style().isEmpty()
    private HtmlTag tryCombineAttrs(
        final HtmlTag tag,
        final List<HtmlNode> nodes)
    {
        HtmlTag result = null;
        boolean combinable = false;
        int size = nodes.size();
        for (int i = 0; i < size; ++i) {
            HtmlNode node = nodes.get(i);
            if (node instanceof HtmlTag) {
                HtmlTag subtag = (HtmlTag) node;
                if (subtag.id() == null
                    && subtag.classes().isEmpty()
                    && subtag.style().isEmpty()
                    && tag.tagName().equals(subtag.tagName()))
                {
                    combinable = true;
                    break;
                }
            }
        }
        if (combinable) {
            result = new HtmlTag(
                HtmlTextEscapingMode.PCDATA,
                "",
                EMPTY_CLASSES,
                EMPTY_CLASSES,
                null,
                null,
                EmptyAttrs.INSTANCE,
                EmptyStyle.INSTANCE);
            for (int i = 0; i < size; ++i) {
                HtmlNode node = nodes.get(i);
                boolean processed = false;
                if (node instanceof HtmlTag) {
                    HtmlTag subtag = (HtmlTag) node;
                    if (subtag.id() == null
                        && subtag.classes().isEmpty()
                        && subtag.style().isEmpty()
                        && tag.tagName().equals(subtag.tagName()))
                    {
                        HtmlTag newTag = new HtmlTag(
                            subtag,
                            tag.escaping(),
                            tag.tagName(),
                            EMPTY_CLASSES,
                            EMPTY_CLASSES,
                            null,
                            null,
                            BasicAttrs.merge(tag.attrs(), subtag.attrs()),
                            tag.style());
                        result.addNode(newTag);
                        processed = true;
                    }
                }
                if (!processed) {
                    HtmlTag newTag = new HtmlTag(
                        tag.escaping(),
                        tag.tagName(),
                        EMPTY_CLASSES,
                        EMPTY_CLASSES,
                        null,
                        null,
                        tag.attrs(),
                        tag.style());
                    newTag.addNode(node);
                    result.addNode(newTag);
                }
            }
        }
        return result;
    }

    private void tryCompactNodes(final List<HtmlNode> nodes) {
        int size = nodes.size();
        int i = 0;
        while (i < size) {
            HtmlNode next = nodes.get(i++);
            if (next instanceof HtmlTag) {
                HtmlTag currentTag = (HtmlTag) next;
                ImmutableTagConfig currentConfig =
                    tags.get(currentTag.tagName());
                if ((currentConfig.combine() || currentConfig.flag())
                    && currentTag.id() == null
                    && currentTag.classes().isEmpty())
                {
                    boolean modified = false;
                    while (i < size) {
                        next = nodes.get(i);
                        if (next instanceof HtmlTag) {
                            HtmlTag nextTag = (HtmlTag) next;
                            if (nextTag.id() == null
                                && nextTag.classes().isEmpty()
                                && currentTag.tagName().equals(
                                    nextTag.tagName())
                                && currentTag.attrs().equals(nextTag.attrs())
                                && currentTag.style().equals(nextTag.style()))
                            {
                                nodes.remove(i);
                                --size;
                                currentTag.addAll(nextTag);
                                modified = true;
                            } else {
                                break;
                            }
                        } else {
                            break;
                        }
                    }
                    if (modified && !currentTag.isEmpty()) {
                        tryCompactNodes(currentTag);
                    }
                }
            }
        }
    }
}

