package ru.yandex.sanitizer2;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.helger.css.ECSSVersion;
import com.helger.css.decl.IHasCSSDeclarations;
import com.helger.css.handler.CSSHandler;
import com.helger.css.parser.ParseException;
import com.helger.css.parser.ParserCSS30;
import com.helger.css.parser.ParserCSS30TokenManager;
import com.helger.css.reader.errorhandler.DoNothingCSSInterpretErrorHandler;
import com.helger.css.reader.errorhandler.ICSSInterpretErrorHandler;
import com.helger.css.reader.errorhandler.ICSSParseErrorHandler;
import com.helger.css.reader.errorhandler.ThrowingCSSParseErrorHandler;
import org.owasp.html.HtmlStreamEventReceiver;
import org.owasp.html.HtmlStreamRenderer;
import org.owasp.html.HtmlTextEscapingMode;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.collection.IntPair;
import ru.yandex.net.uri.fast.FastUri;
import ru.yandex.net.uri.fast.FastUriParser;
import ru.yandex.sanitizer2.config.ImmutablePropertyConfig;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.sanitizer2.config.ImmutableTagConfig;

public class HtmlDomBuilder implements HtmlStreamEventReceiver {
    private static final ICSSParseErrorHandler PARSE_ERROR_HANDLER =
        new ThrowingCSSParseErrorHandler();
    private static final ICSSInterpretErrorHandler INTERPRETER_ERROR_HANDLER =
        new DoNothingCSSInterpretErrorHandler();
    private static final int INITIAL_STACK_CAPACITY = 16;
    private static final char[] EMPTY_BUF = new char[0];

    private final Map<String, IntPair<Style>> styleCache = new HashMap<>();
    private final HtmlTag root = new HtmlTag(
        HtmlTextEscapingMode.PCDATA,
        "",
        Collections.emptyList(),
        Collections.emptyList(),
        null,
        null,
        EmptyAttrs.INSTANCE,
        EmptyStyle.INSTANCE);
    private final SanitizingContext context = new SanitizingContext();
    private final List<String> tmpClasses = new ArrayList<>(2);
    private final BasicAttrs tmpAttrs = new BasicAttrs(4);
    private final ImmutableSanitizingConfig config;
    private final StyleExtractor styleExtractor;
    private final String styleTag;
    private final String baseTag;
    private final String classAttr;
    private final String idAttr;
    private final String styleAttr;
    private final String hrefAttr;
    private final ImmutablePropertyConfig classConfig;
    private final ImmutablePropertyConfig idConfig;
    private List<CssDeclaration> tmpDeclarations = null;
    private HtmlCDataTag styleBlock = null;
    private AbstractHtmlTag[] stack =
        new AbstractHtmlTag[INITIAL_STACK_CAPACITY];
    private int stackSize = 1;
    private char[] tmpCharBuf = EMPTY_BUF;

    public HtmlDomBuilder(final ImmutableSanitizingConfig config) {
        this.config = config;
        styleExtractor = new StyleExtractor(config, context);
        styleTag = config.styleTag();
        baseTag = config.baseTag();
        classAttr = config.classAttr();
        idAttr = config.idAttr();
        styleAttr = config.styleAttr();
        hrefAttr = config.hrefAttr();
        classConfig = config.classConfig();
        idConfig = config.idConfig();
        stack[0] = root;
    }

    public HtmlTag root() {
        return root;
    }

    public StyleProcessor styleProcessor() {
        return styleExtractor.styleProcessor();
    }

    @Override
    public void openDocument() {
    }

    @Override
    public void closeDocument() {
    }

    // Attributes will be already sanitized by this point, so not duplicate
    // attributes will present. Also all attributes will be in lower-case.
    @Override
    @SuppressWarnings("ReferenceEquality")
    public void openTag(
        final String tagName,
        final List<String> attrsList)
    {
        if (styleBlock != null) {
            // No nested tags allowed inside style block
            return;
        }
        if (HtmlStreamRenderer.isValidHtmlName(tagName)) {
            HtmlTextEscapingMode escaping =
                HtmlTextEscapingMode.getModeForTag(tagName);
            Attrs attrs;
            List<String> classes = Collections.emptyList();
            tmpClasses.clear();
            String id = null;
            Style style = EmptyStyle.INSTANCE;
            int attrsSize = attrsList.size();
            if (attrsSize == 0) {
                attrs = EmptyAttrs.INSTANCE;
            } else {
                tmpAttrs.clear();
                ImmutableTagConfig tagConfig = config.tags().get(tagName);
                for (int i = 0; i < attrsSize;) {
                    // intern result will never be null, because all other
                    // attrs will be intercepted by policy
                    String name = config.internAttr(attrsList.get(i++));
                    String value = attrsList.get(i++);
                    if (name == classAttr) {
                        extractClasses(value);
                    } else if (name == idAttr) {
                        id = value;
                    } else if (name == styleAttr) {
                        style = parseStyle(value);
                    } else if (HtmlStreamRenderer.isValidHtmlName(name)) {
                        ImmutablePropertyConfig attrConfig =
                            tagConfig.attrs().get(name);
                        if (attrConfig == null) {
                            attrConfig = config.globalAttrs().get(name);
                        }
                        String sanitizedValue =
                            attrConfig.sanitize(value, context);
                        if (sanitizedValue != null) {
                            tmpAttrs.put(
                                name,
                                new Attr(sanitizedValue, attrConfig));
                        }
                    }
                }
                attrs = tmpAttrs.compact();
                if (attrs == tmpAttrs) {
                    attrs = new BasicAttrs(tmpAttrs);
                }
            }
            String sanitizedId;
            if (id == null) {
                sanitizedId = null;
            } else {
                sanitizedId = idConfig.sanitize(id, context);
            }
            List<String> sanitizedClasses;
            int classesCount = tmpClasses.size();
            if (classesCount == 0) {
                sanitizedClasses = classes;
            } else {
                classes = CollectionCompactor.compactOrCopy(tmpClasses);
                for (int i = 0; i < classesCount; ++i) {
                    tmpClasses.set(
                        i,
                        classConfig.sanitize(tmpClasses.get(i), context));
                }
                sanitizedClasses =
                    CollectionCompactor.compactOrCopy(tmpClasses);
            }
            if (tagName == styleTag) {
                styleBlock = new HtmlCDataTag(
                    escaping,
                    tagName,
                    classes,
                    sanitizedClasses,
                    id,
                    sanitizedId,
                    attrs,
                    style);
            } else if (tagName == baseTag) {
                Attr href = attrs.get(hrefAttr);
                if (href != null) {
                    String value = href.value();
                    if (!value.isEmpty()) {
                        try {
                            FastUri uri = new FastUriParser(value).parse();
                            if (uri.scheme() != null && uri.path() != null) {
                                uri = uri.normalize();
                                if (uri.path().isEmpty()) {
                                    uri.path("/");
                                }
                                context.baseHref(uri);
                            }
                        } catch (URISyntaxException e) {
                            // ignore
                        }
                    }
                }
            } else {
                AbstractHtmlTag tag;
                if (HtmlStreamRenderer.unescaped(escaping)) {
                    tag = new HtmlCDataTag(
                        escaping,
                        tagName,
                        classes,
                        sanitizedClasses,
                        id,
                        sanitizedId,
                        attrs,
                        style);
                } else {
                    tag = new HtmlTag(
                        escaping,
                        tagName,
                        classes,
                        sanitizedClasses,
                        id,
                        sanitizedId,
                        attrs,
                        style);
                }
                if (stack[stackSize - 1].addNode(tag)
                    && escaping != HtmlTextEscapingMode.VOID)
                {
                    // Check that there is place for spare element in stack
                    if (stackSize + 1 >= stack.length) {
                        stack = Arrays.copyOf(stack, stackSize << 1);
                    }
                    // Add was successful and it could contain subtags,
                    // so closeTag will be called, add it to stack
                    stack[stackSize++] = tag;
                }
            }
        }
    }

    private Style parseStyle(final String value) {
        IntPair<Style> cached = styleCache.get(value);
        int contextGeneration = context.contextGeneration();
        Style style;
        if (cached == null || cached.first() != contextGeneration) {
            style = EmptyStyle.INSTANCE;
            IHasCSSDeclarations<?> css;
            try {
                ParserCSS30TokenManager tokenManager =
                    new ParserCSS30TokenManager(new StringCharStream(value));
                tokenManager.setCustomErrorHandler(PARSE_ERROR_HANDLER);
                ParserCSS30 parser = new ParserCSS30(tokenManager);
                parser.setBrowserCompliantMode(true);
                parser.setCustomErrorHandler(PARSE_ERROR_HANDLER);
                css = CSSHandler.readDeclarationListFromNode(
                    ECSSVersion.CSS30,
                    INTERPRETER_ERROR_HANDLER,
                    false,
                    parser.styleDeclarationList());
            } catch (RuntimeException | ParseException e) {
                css = null;
            }
            if (css != null && css.getDeclarationCount() > 0) {
                if (tmpDeclarations == null) {
                    tmpDeclarations =
                        new ArrayList<>(css.getDeclarationCount());
                } else {
                    tmpDeclarations.clear();
                }
                BasicStyle.convertDeclarations(
                    tmpDeclarations,
                    config,
                    css,
                    context);
                if (!tmpDeclarations.isEmpty()) {
                    style = new BasicStyle(tmpDeclarations).compact();
                }
            }
            styleCache.put(value, new IntPair<>(contextGeneration, style));
        } else {
            style = cached.second();
        }
        return style;
    }

    @Override
    public void closeTag(final String tagName) {
        if (styleBlock == null) {
            // check if we should ignore erroneous tag
            if (HtmlStreamRenderer.isValidHtmlName(tagName)
                && tagName.equals(stack[stackSize - 1].tagName()))
            {
                --stackSize;
            }
        } else if (styleTag.equals(tagName)) {
            styleExtractor.parseStyleBlock(styleBlock);
            styleBlock = null;
        }
    }

    @Override
    public void text(final String text) {
        if (styleBlock == null) {
            stack[stackSize - 1].addTextNode(text);
        } else {
            styleBlock.addTextNode(text);
        }
    }

    private static int nextSpace(
        final char[] buf,
        final int len,
        final int startPos)
    {
        for (int i = startPos; i < len; ++i) {
            if (Character.isWhitespace(buf[i])) {
                return i;
            }
        }
        return -1;
    }

    private void extractClasses(final String classes) {
        tmpClasses.clear();
        int len = classes.length();
        if (len > tmpCharBuf.length) {
            tmpCharBuf = new char[Math.max(len, tmpCharBuf.length << 1)];
        }
        classes.getChars(0, len, tmpCharBuf, 0);
        int idx = nextSpace(tmpCharBuf, len, 0);
        if (idx == -1) {
            tmpClasses.add(classes);
        } else {
            if (idx > 0) {
                tmpClasses.add(classes.substring(0, idx));
            }
            int pos = idx + 1;
            while (pos < len) {
                idx = nextSpace(tmpCharBuf, len, pos);
                if (idx == -1) {
                    idx = len;
                }
                int classLen = idx - pos;
                if (classLen > 0) {
                    tmpClasses.add(classes.substring(pos, pos + classLen));
                }
                pos = idx + 1;
            }
        }
    }
}

