package ru.yandex.sanitizer2;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import com.helger.css.CCSS;
import org.owasp.html.Encoding;
import org.owasp.html.HtmlStreamRenderer;
import org.owasp.html.HtmlTextEscapingMode;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.sanitizer2.config.ImmutablePropertyConfig;
import ru.yandex.sanitizer2.config.ImmutablePropertyFoldConfig;
import ru.yandex.sanitizer2.config.ImmutablePropertyFoldsConfig;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.sanitizer2.config.ImmutableUrlConfig;

public class HtmlPrinter<E extends Exception>
    implements HtmlNodeVisitor<Void, E>
{
    // Temporary list for attr urls detection
    private final List<AttrUrlInfo> attrUrls = new ArrayList<>();
    // Css and contained url rewriting is pretty heavy, so cache it
    private final Map<Style, AttrWithUrls> rewrittenStyleCache =
        new HashMap<>();
    private final StringBuilder tmpSb = new StringBuilder();
    // Config for tags renaming
    private final ImmutableSanitizingConfig config;
    private final HtmlCollector<E> htmlCollector;
    private final UrlCollector urlCollector;
    private final AttrPostProcessor attrPostProcessor;
    private final CssPostProcessor cssPostProcessor;
    // Temporary list for flushed urls
    private List<AttrUrlInfo> flushedUrls = null;
    // Accumulates suffixes that should be appended to data in reverse order
    private List<String> suffixes = null;
    private boolean done = false;

    public HtmlPrinter(
        final ImmutableSanitizingConfig config,
        final HtmlCollector<E> htmlCollector,
        final UrlCollector urlCollector,
        final AttrPostProcessor attrPostProcessor,
        final CssPostProcessor cssPostProcessor)
    {
        this.config = config;
        this.htmlCollector = htmlCollector;
        this.urlCollector = urlCollector;
        this.attrPostProcessor = attrPostProcessor;
        this.cssPostProcessor = cssPostProcessor;
    }

    public void done() throws E {
        if (!done) {
            done = true;
            if (suffixes != null) {
                int size = suffixes.size();
                for (int i = size - 1; i >= 0; --i) {
                    htmlCollector.append(suffixes.get(i));
                }
            }
            htmlCollector.flush();
        }
    }

    public void append(final String text) {
        htmlCollector.append(text);
    }

    public void appendSuffix(final String suffix) {
        if (!suffix.isEmpty()) {
            if (suffixes == null) {
                suffixes = new ArrayList<>(2);
            }
            suffixes.add(suffix);
        }
    }

    private void closeAttrValue(final String attrValue) {
        // Workaround IE8. Prevent double quotes omitting when copying
        // attribute value using js, which causes <div attr="``foo=bar"> become
        // <div attr=``foo=bar>, which will be treated by IE8 as
        // <div attr="" foo="bar">.
        if (attrValue.indexOf('`') != -1) {
            htmlCollector.append(' ');
        }
        htmlCollector.append('"');
    }

    private void writeAttrValue(final String value) {
        if (config.preserveDoubleBraces()) {
            Encoding.encodeHtmlOnto(
                value,
                htmlCollector.sb(),
                null,
                Encoding.REPLACEMENTS);
        } else {
            Encoding.encodeHtmlAttribOnto(value, htmlCollector.sb());
        }
    }

    private void writeAttr(final String attrName, final AttrWithUrls attr)
        throws E
    {
        String attrValue = attr.attrValue();
        if (!attrValue.isEmpty()) {
            List<AttrUrlInfo> urls = attr.urls();
            if (urls.isEmpty()) {
                htmlCollector.append(' ');
                htmlCollector.append(attrName);
                htmlCollector.append('=');
                htmlCollector.append('"');
                writeAttrValue(attrValue);
                closeAttrValue(attrValue);
            } else {
                htmlCollector.append(' ');
                htmlCollector.flush();
                int attrStart = htmlCollector.length();
                htmlCollector.append(attrName);
                htmlCollector.append('=');
                htmlCollector.append('"');
                int size = urls.size();
                if (flushedUrls == null) {
                    flushedUrls = new ArrayList<>(size);
                } else {
                    flushedUrls.clear();
                }
                int prevPos = 0;
                for (int i = 0; i < size; ++i) {
                    AttrUrlInfo url = urls.get(i);
                    int urlStart = url.urlStart();
                    int urlEnd = url.urlEnd();
                    String value = attrValue.substring(prevPos, urlStart);
                    writeAttrValue(value);
                    htmlCollector.flush();
                    int dataStart = htmlCollector.length();
                    value = attrValue.substring(urlStart, urlEnd);
                    writeAttrValue(value);
                    htmlCollector.flush();
                    flushedUrls.add(
                        new AttrUrlInfo(
                            dataStart,
                            htmlCollector.length(),
                            url.urlClassId(),
                            url.urlClassValue()));
                    prevPos = urlEnd;
                }
                String value = attrValue.substring(prevPos);
                writeAttrValue(value);
                closeAttrValue(attrValue);
                htmlCollector.flush();
                int attrLen = htmlCollector.length() - attrStart;
                for (int i = 0; i < size; ++i) {
                    urlCollector.add(attrStart, attrLen, flushedUrls.get(i));
                }
            }
        }
    }

    private void openTag(final AbstractHtmlTag tag, final boolean empty)
        throws E
    {
        String tagName = tag.tagName();
        htmlCollector.append('<');
        String renameTo = config.tags().get(tagName).renameTo();
        if (renameTo == null) {
            htmlCollector.append(tagName);
        } else {
            htmlCollector.append(renameTo);
        }
        List<String> classes = tag.sanitizedClasses();
        int classesCount = classes.size();
        if (classesCount != 0) {
            boolean hasApos = false;
            htmlCollector.append(" class=\"");
            for (int i = 0; i < classesCount; ++i) {
                if (i != 0) {
                    htmlCollector.append(' ');
                }
                String className = classes.get(i);
                writeAttrValue(className);
                if (!hasApos && className.indexOf('`') != -1) {
                    hasApos = true;
                }
            }
            // see closeAttrValue(String)
            if (hasApos) {
                htmlCollector.append(' ');
            }
            htmlCollector.append('"');
        }
        String id = tag.sanitizedId();
        if (id != null) {
            htmlCollector.append(" id=\"");
            writeAttrValue(id);
            closeAttrValue(id);
        }
        Attrs attrs = tag.attrs();
        if (!attrs.isEmpty()) {
            for (Map.Entry<String, Attr> entry: sorted(attrs).entrySet()) {
                String attrName = entry.getKey();
                Attr attr = entry.getValue();
                String attrValue = attr.value();
                ImmutablePropertyConfig attrConfig = attr.config();
                ImmutableUrlConfig urlConfig = attrConfig.urlConfig();
                attrUrls.clear();
                if (urlConfig != null) {
                    attrUrls.add(
                        urlConfig.type().detectUrlClass(
                            0,
                            attrValue.length(),
                            attrValue,
                            config));
                }
                AttrWithUrls attrWithUrls = attrPostProcessor.process(
                    tagName,
                    attrName,
                    new AttrWithUrls(attrValue, attrUrls),
                    attrConfig);
                if (attrWithUrls != null) {
                    writeAttr(attrName, attrWithUrls);
                }
            }
        }
        Style style = cssPostProcess(tag.style());
        if (!style.isEmpty()) {
            AttrWithUrls attr = rewrittenStyleCache.get(style);
            if (attr == null) {
                Style folded = fold(style);
                attrUrls.clear();
                tmpSb.setLength(0);
                for (CssDeclaration declaration: sorted(folded).values()) {
                    if (tmpSb.length() > 0) {
                        tmpSb.append(CCSS.DEFINITION_END);
                    }
                    tmpSb.append(declaration.property());
                    tmpSb.append(CCSS.SEPARATOR_PROPERTY_VALUE);
                    int off = tmpSb.length();
                    List<AttrUrlInfo> urls = declaration.urls();
                    int size = urls.size();
                    for (int i = 0; i < size; ++i) {
                        AttrUrlInfo url = urls.get(i);
                        attrUrls.add(
                            new AttrUrlInfo(
                                url.urlStart() + off,
                                url.urlEnd() + off,
                                url.urlClassId(),
                                url.urlClassValue()));
                    }
                    tmpSb.append(declaration.value());
                    if (declaration.important()) {
                        tmpSb.append(CCSS.IMPORTANT_SUFFIX);
                    }
                }
                attr = new AttrWithUrls(
                    tmpSb.toString(),
                    CollectionCompactor.compactOrCopy(attrUrls));
                rewrittenStyleCache.put(style, attr);
            }
            writeAttr("style", attr);
        }

        if (empty) {
            htmlCollector.append(' ');
            htmlCollector.append('/');
        }
        htmlCollector.appendAndFlush('>');
    }

    private void closeTag(final String tagName) throws E {
        htmlCollector.append('<');
        htmlCollector.append('/');
        String renameTo = config.tags().get(tagName).renameTo();
        if (renameTo == null) {
            htmlCollector.append(tagName);
        } else {
            htmlCollector.append(renameTo);
        }
        htmlCollector.appendAndFlush('>');
    }

    @Override
    public Void visit(final HtmlTag tag) throws E {
        String tagName = tag.tagName();
        if (tagName.isEmpty()) {
            int size = tag.size();
            for (int i = 0; i < size; ++i) {
                tag.get(i).accept(this);
            }
        } else {
            boolean empty = tag.escaping() == HtmlTextEscapingMode.VOID;
            openTag(tag, empty);
            if (!empty) {
                int size = tag.size();
                for (int i = 0; i < size; ++i) {
                    tag.get(i).accept(this);
                }
                closeTag(tagName);
            }
        }
        return null;
    }

    @Override
    public Void visit(final HtmlCDataTag cdata) throws E {
        String tagName = cdata.tagName();
        openTag(cdata, false);
        StringBuilder text = cdata.text();
        Encoding.stripBannedCodeunits(text);
        // Check if data is valid, skip otherwise
        if (HtmlStreamRenderer.checkHtmlCdataCloseable(tagName, text) == -1) {
            htmlCollector.append(text);
        }
        closeTag(tagName);
        return null;
    }

    @Override
    public Void visit(final HtmlText text) {
        String value = text.text();
        if (text.escaping() == HtmlTextEscapingMode.RCDATA) {
            if (config.preserveDoubleBraces()) {
                Encoding.encodeHtmlOnto(
                    value,
                    htmlCollector.sb(),
                    null,
                    Encoding.TEXT_REPLACEMENTS);
            } else {
                Encoding.encodeRcdataOnto(value, htmlCollector.sb());
            }
        } else {
            if (config.preserveDoubleBraces()) {
                Encoding.encodeHtmlOnto(
                    value,
                    htmlCollector.sb(),
                    null,
                    Encoding.TEXT_REPLACEMENTS);
            } else {
                Encoding.encodePcdataOnto(value, htmlCollector.sb());
            }
        }
        return null;
    }

    private static <K, V> Map<K, V> sorted(final Map<K, V> map) {
        if (map.size() > 1) {
            return new TreeMap<>(map);
        } else {
            return map;
        }
    }

    private static boolean containsChar(final String s, final char c) {
        int len = s.length();
        char closeChar = 0;
        for (int i = 0; i < len; ++i) {
            char current = s.charAt(i);
            if (closeChar == 0) {
                switch (current) {
                    case '\'':
                        closeChar = '\'';
                        break;
                    case '"':
                        closeChar = '"';
                        break;
                    case '(':
                        closeChar = ')';
                        break;
                    default:
                        if (current == c) {
                            return true;
                        }
                        break;
                }
            } else if (current == closeChar) {
                closeChar = 0;
            }
        }
        return false;
    }

    private static boolean nonFoldable(
        final CssDeclaration declaration,
        final ImmutableSanitizingConfig.Fold fold,
        final Style style)
    {
        if (!declaration.important()) {
            String value = declaration.value();
            if ((fold.allowSpaces() || !containsChar(value, ' '))
                && (fold.allowCommas() || !containsChar(value, ','))
                && (value.length() != 7
                    || (!"initial".equals(value) && !"inherit".equals(value))))
            {
                String requiredProperty = fold.requiredProperty();
                if (requiredProperty == null
                    || style.containsKey(requiredProperty))
                {
                    return false;
                }
            }
        }
        return true;
    }

    private Set<String> foldableProperties(final Style style) {
        Map<String, int[]> foldableProperties = null;
        for (CssDeclaration declaration: style.values()) {
            ImmutableSanitizingConfig.Fold fold =
                config.folds().get(declaration.property());
            if (fold != null) {
                String shorthandName = fold.shorthandName();
                if (foldableProperties == null) {
                    foldableProperties = new HashMap<>();
                    int mandatoriesLeft;
                    if (nonFoldable(declaration, fold, style)) {
                        // This property shouldn't be folded
                        mandatoriesLeft = -1;
                    } else {
                        mandatoriesLeft =
                            fold.mandatoryPropertiesCount();
                        if (fold.mandatory()) {
                            --mandatoriesLeft;
                        }
                    }
                    foldableProperties.put(
                        shorthandName,
                        new int[]{mandatoriesLeft});
                } else if (nonFoldable(declaration, fold, style)) {
                    // Whatever it was, it shouldn't be folded
                    foldableProperties.put(
                        shorthandName,
                        new int[]{-1});
                } else {
                    int[] oldCounter =
                        foldableProperties.get(shorthandName);
                    if (oldCounter == null) {
                        int mandatoriesLeft =
                            fold.mandatoryPropertiesCount();
                        if (fold.mandatory()) {
                            --mandatoriesLeft;
                        }
                        foldableProperties.put(
                            shorthandName,
                            new int[]{mandatoriesLeft});
                    } else if (fold.mandatory()) {
                        --oldCounter[0];
                    }
                }
            }
        }
        if (foldableProperties != null) {
            Iterator<Map.Entry<String, int[]>> iter =
                foldableProperties.entrySet().iterator();
            do {
                Map.Entry<String, int[]> entry = iter.next();
                if (entry.getValue()[0] != 0
                    || style.containsKey(entry.getKey()))
                {
                    iter.remove();
                }
            } while (iter.hasNext());
        }
        if (foldableProperties == null || foldableProperties.isEmpty()) {
            return null;
        } else {
            return foldableProperties.keySet();
        }
    }

    private void writeValue(
        final CssDeclaration declaration,
        final ImmutablePropertyFoldConfig foldConfig)
    {
        if (tmpSb.length() > 0) {
            tmpSb.append(foldConfig.separator());
        }
        int off = tmpSb.length();
        tmpSb.append(declaration.value());
        List<AttrUrlInfo> urls = declaration.urls();
        int size = urls.size();
        for (int i = 0; i < size; ++i) {
            AttrUrlInfo url = urls.get(i);
            attrUrls.add(
                new AttrUrlInfo(
                    url.urlStart() + off,
                    url.urlEnd() + off,
                    url.urlClassId(),
                    url.urlClassValue()));
        }
    }

    private Style cssPostProcess(final Style style) {
        int size = style.size();
        switch (size) {
            case 0:
                return style;
            case 1:
                {
                    CssDeclaration declaration =
                        style.values().iterator().next();
                    CssDeclaration processed =
                        cssPostProcessor.process(declaration);
                    if (processed == null) {
                        return EmptyStyle.INSTANCE;
                    } else if (processed == declaration) {
                        return style;
                    } else {
                        return new SingletonStyle(processed);
                    }
                }
            default:
                BasicStyle newStyle = new BasicStyle();
                for (CssDeclaration declaration: style.values()) {
                    CssDeclaration processed =
                        cssPostProcessor.process(declaration);
                    if (processed != null) {
                        newStyle.put(processed.property(), processed);
                    }
                }
                return newStyle.compact();
        }
    }

    private void foldProperty(final BasicStyle style, final String property) {
        ImmutablePropertyFoldsConfig foldsConfig =
            config.shorthands().get(property).foldsConfig();
        boolean foldAllEqual = foldsConfig.foldAllEqual();
        tmpSb.setLength(0);
        attrUrls.clear();
        CssDeclaration skippedDeclaration = null;
        ImmutablePropertyFoldConfig skippedFoldConfig = null;
        String defaultValue = null;
        List<AttrUrlInfo> defaultUrls = null;
        String allEqualValue = null;
        List<AttrUrlInfo> allEqualUrls = null;
        for (Map.Entry<String, ImmutablePropertyFoldConfig> entry
            : foldsConfig.properties().entrySet())
        {
            CssDeclaration declaration =
                style.remove(entry.getKey());
            if (declaration != null) {
                ImmutablePropertyFoldConfig foldConfig = entry.getValue();
                String value = declaration.value();
                if (!value.equals(foldConfig.defaultValue())) {
                    if (skippedDeclaration != null) {
                        if (foldConfig.requiredProperty() != null) {
                            writeValue(skippedDeclaration, skippedFoldConfig);
                        }
                        skippedDeclaration = null;
                    }
                    writeValue(declaration, foldConfig);
                } else {
                    skippedDeclaration = declaration;
                    skippedFoldConfig = foldConfig;
                    if (defaultValue == null) {
                        defaultValue = value;
                        defaultUrls = declaration.urls();
                    }
                }
                if (foldAllEqual) {
                    if (allEqualValue == null) {
                        allEqualValue = value;
                        allEqualUrls = declaration.urls();
                    } else if (!allEqualValue.equals(value)) {
                        allEqualValue = null;
                        foldAllEqual = false;
                    }
                }
            }
        }
        String value;
        List<AttrUrlInfo> urls;
        if (allEqualValue == null) {
            if (tmpSb.length() > 0) {
                value = tmpSb.toString();
                urls = CollectionCompactor.compactOrCopy(attrUrls);
            } else {
                value = defaultValue;
                urls = defaultUrls;
            }
        } else {
            value = allEqualValue;
            urls = allEqualUrls;
        }
        style.put(
            property,
            new CssDeclaration(property, value, false, null, urls));
    }

    private Style fold(final Style style) {
        Set<String> foldableProperties;
        if (config.foldProperties()) {
            foldableProperties = foldableProperties(style);
        } else {
            foldableProperties = null;
        }
        if (foldableProperties == null) {
            return style;
        } else {
            BasicStyle folded = new BasicStyle(style);
            for (String property: foldableProperties) {
                foldProperty(folded, property);
            }
            return fold(folded);
        }
    }
}

