package ru.yandex.sanitizer2;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import ru.yandex.function.CharArrayProcessable;
import ru.yandex.jni.pcre.JniPcre;
import ru.yandex.jni.pcre.JniPcreException;
import ru.yandex.sanitizer2.config.ImmutablePageHeaderConfig;

public class MultiPageHeaderApplier implements PageHeaderApplier {
    private final Consumer<String> pageHeadersAccountant;
    private final List<PageHeaderInfo> substringConfigs;
    private final Map<Integer, PageHeaderInfo> jniConfigs;
    private final JniPcre jniPcre;

    public MultiPageHeaderApplier(
        final Consumer<String> pageHeadersAccountant)
        throws PageHeaderException
    {
        this.pageHeadersAccountant = pageHeadersAccountant;
        substringConfigs = new ArrayList<>();
        jniConfigs = new HashMap<>();
        try {
            jniPcre = new JniPcre(true);
        } catch (JniPcreException e) {
            throw new PageHeaderException("Failed to initialize JniPcre", e);
        }
    }

    public void addSubstringPredicate(
        final String name,
        final ImmutablePageHeaderConfig config)
    {
        substringConfigs.add(new PageHeaderInfo(name, config));
    }

    public void addRegexPredicate(
        final String name,
        final ImmutablePageHeaderConfig config)
        throws PageHeaderException
    {
        try {
            int id = jniPcre.addRegex(config.content());
            jniConfigs.put(id, new PageHeaderInfo(name, config));
        } catch (JniPcreException e) {
            throw new PageHeaderException("Failed to add regex", e);
        }
    }

    private PageHeaderInfo findMatch(final String text)
        throws PageHeaderException
    {
        for (int i = 0; i < substringConfigs.size(); ++i) {
            PageHeaderInfo info = substringConfigs.get(i);
            if (text.contains(info.config.content())) {
                return info;
            }
        }
        try {
            int i = jniPcre.findMatch(text);
            if (i >= 0) {
                return jniConfigs.get(i);
            }
        } catch (JniPcreException e) {
            throw new PageHeaderException("Match failed", e);
        }
        return null;
    }

    @Override
    public void accept(
        final CharArrayProcessable text,
        final HtmlPrinter<?> printer)
        throws PageHeaderException
    {
        String body = text.toString();
        PageHeaderInfo info = findMatch(body);
        if (info != null) {
            printer.append(info.config.header());
            printer.appendSuffix(info.config.footer());
            pageHeadersAccountant.accept(info.name);
        }
    }

    @Override
    public void close() {
        jniPcre.close();
    }

    public PageHeaderApplier compact() {
        PageHeaderApplier result = this;
        if (substringConfigs.isEmpty() && jniConfigs.isEmpty()) {
            result = EmptyPageHeaderApplier.INSTANCE;
        }
        if (result != this) {
            this.close();
        }
        return result;
    }

    private static class PageHeaderInfo {
        private final String name;
        private final ImmutablePageHeaderConfig config;

        public PageHeaderInfo(
            final String name,
            final ImmutablePageHeaderConfig config)
        {
            this.name = name;
            this.config = config;
        }
    }
}

