package ru.yandex.sanitizer2;

import java.util.Collections;
import java.util.List;

public class PgpSignatureFoldingVisitor
    implements HtmlNodeVisitor<Void, RuntimeException>
{
    private static final String PGP_SIGNED_MESSAGE_START =
        "-----BEGIN PGP SIGNED MESSAGE-----";
    private static final String PGP_SIGNATURE_START =
        "-----BEGIN PGP SIGNATURE-----";
    private static final String PGP_SIGNATURE_END =
        "-----END PGP SIGNATURE-----";

    private final String blockquoteTag;

    public PgpSignatureFoldingVisitor(final String blockquoteTag) {
        this.blockquoteTag = blockquoteTag;
    }

    @Override
    public Void visit(final HtmlTag node) {
        int size = node.size();
        if (size > 0) {
            for (int i = 0; i < size; ++i) {
                node.get(i).accept(this);
            }
            foldPgpSignature(node);
        }
        return null;
    }

    @Override
    public Void visit(final HtmlCDataTag node) {
        return null;
    }

    @Override
    public Void visit(final HtmlText node) {
        return null;
    }

    private void foldPgpSignature(final List<HtmlNode> nodes) {
        int pos = 0;
        while (pos < nodes.size()) {
            int messageStart =
                findTextNode(nodes, PGP_SIGNED_MESSAGE_START, pos);
            if (messageStart == -1) {
                break;
            }
            int emptyLine = findEmptyTextNode(nodes, messageStart + 1);
            if (emptyLine == -1) {
                break;
            }
            int signatureStart =
                findTextNode(nodes, PGP_SIGNATURE_START, emptyLine + 1);
            if (signatureStart == -1) {
                break;
            }
            int signatureEnd =
                findTextNode(nodes, PGP_SIGNATURE_END, signatureStart + 1);
            if (signatureEnd == -1) {
                break;
            }
            HtmlText messageStartText = (HtmlText) nodes.get(messageStart);
            int idx = messageStartText.text().indexOf(PGP_SIGNED_MESSAGE_START);
            String prefix = messageStartText.text().substring(0, idx);
            if (!trim(prefix).isEmpty()) {
                nodes.set(
                    messageStart,
                    new HtmlText(messageStartText.escaping(), prefix));
                ++messageStart;
            }

            HtmlText signatureStartText = (HtmlText) nodes.get(signatureStart);
            idx = signatureStartText.text().indexOf(PGP_SIGNATURE_START);
            prefix = signatureStartText.text().substring(0, idx);
            if (!trim(prefix).isEmpty()) {
                nodes.set(
                    signatureStart,
                    new HtmlText(
                        signatureStartText.escaping(),
                        signatureStartText.text().substring(idx)));
                nodes.add(
                    signatureStart,
                    new HtmlText(
                        signatureStartText.escaping(),
                        prefix));
                ++signatureStart;
                ++signatureEnd;
            }

            HtmlText signatureEndText = (HtmlText) nodes.get(signatureEnd);
            idx =
                signatureEndText.text().indexOf(PGP_SIGNATURE_END)
                + PGP_SIGNATURE_END.length();
            String suffix = signatureEndText.text().substring(idx);
            if (!trim(suffix).isEmpty()) {
                nodes.set(
                    signatureEnd,
                    new HtmlText(
                        signatureEndText.escaping(),
                        signatureEndText.text().substring(0, idx)));
                nodes.add(
                    signatureEnd + 1,
                    new HtmlText(
                        signatureEndText.escaping(),
                        suffix));
            }

            List<HtmlNode> signatureNodes =
                nodes.subList(
                    signatureStart,
                    captureBr(nodes, signatureEnd) + 1);
            List<String> classes = Collections.singletonList("wmi-pgp");
            HtmlTag signatureNode = new HtmlTag(
                signatureNodes,
                signatureStartText.escaping(),
                blockquoteTag,
                classes,
                classes,
                null,
                null,
                EmptyAttrs.INSTANCE,
                EmptyStyle.INSTANCE);
            signatureNodes.clear();
            nodes.add(signatureStart, signatureNode);
            nodes.subList(
                messageStart,
                captureBr(nodes, emptyLine) + 1).clear();
            pos = messageStart + 1;
        }
    }

    private static int captureBr(
        final List<HtmlNode> nodes,
        final int pos)
    {
        if (pos + 1 < nodes.size()) {
            HtmlNode next = nodes.get(pos + 1);
            if (next instanceof HtmlTag
                && ((HtmlTag) next).tagName().equals("br"))
            {
                return pos + 1;
            }
        }
        return pos;
    }

    private static int findTextNode(
        final List<HtmlNode> nodes,
        final String text,
        final int fromIndex)
    {
        int size = nodes.size();
        for (int i = fromIndex; i < size; ++i) {
            HtmlNode node = nodes.get(i);
            if (node instanceof HtmlText
                && ((HtmlText) node).text().indexOf(text) != -1)
            {
                return i;
            }
        }
        return -1;
    }

    private static int findEmptyTextNode(
        final List<HtmlNode> nodes,
        final int fromIndex)
    {
        int size = nodes.size();
        for (int i = fromIndex; i < size; ++i) {
            HtmlNode node = nodes.get(i);
            if (node instanceof HtmlText
                && ((HtmlText) node).text().isEmpty())
            {
                return i;
            }
        }
        return -1;
    }

    private static String trim(final String text) {
        int len = text.length();
        int start = 0;
        int end = len;
        while (start < end) {
            char c = text.charAt(start);
            if (c <= ' ' || Character.isSpaceChar(c)) {
                ++start;
            } else {
                break;
            }
        }
        while (end > start) {
            char c = text.charAt(end - 1);
            if (c <= ' ' || Character.isSpaceChar(c)) {
                --end;
            } else {
                break;
            }
        }
        if (start > 0 || end < len) {
            return text.substring(start, end);
        } else {
            return text;
        }
    }
}

