package ru.yandex.sanitizer2;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.owasp.html.HtmlTextEscapingMode;

import ru.yandex.net.uri.fast.FastUri;
import ru.yandex.sanitizer2.config.ImmutablePropertyConfig;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.url.processor.StringUrlsExtractor;
import ru.yandex.url.processor.UrlInfo;

public class PlainLinksWrappingVisitor
    implements HtmlNodeVisitor<Void, RuntimeException>
{
    private final String aTag;
    private final String hrefAttr;
    private final ImmutablePropertyConfig hrefConfig;
    private final StringUrlsExtractor urlsExtractor;

    public PlainLinksWrappingVisitor(
        final ImmutableSanitizingConfig config)
    {
        aTag = config.aTag();
        hrefAttr = config.hrefAttr();
        hrefConfig = config.tags().get(aTag).attrs().get(hrefAttr);
        urlsExtractor = new StringUrlsExtractor(
            config.urlSanitizingConfig().urlProcessorConfig());
    }

    @Override
    @SuppressWarnings("ReferenceEquality")
    public Void visit(final HtmlTag node) {
        int size = node.size();
        if (size > 0
            && node.tagName() != aTag
            && hrefConfig != null)
        {
            for (int i = 0; i < size; ++i) {
                node.get(i).accept(this);
            }
            wrapPlainLinks(node.escaping(), node);
        }
        return null;
    }

    @Override
    public Void visit(final HtmlCDataTag node) {
        return null;
    }

    @Override
    public Void visit(final HtmlText node) {
        return null;
    }

    private void wrapPlainLinks(
        final HtmlTextEscapingMode escaping,
        final List<HtmlNode> nodes)
    {
        int pos = 0;
        while (pos < nodes.size()) {
            int textStart = findTextNode(nodes, pos);
            if (textStart == -1) {
                break;
            }
            // will fold consequent text nodes, so we can jump to next non-text
            // node and after that step on the next node
            pos = textStart + 2;
            int textEnd = findNonTextNode(nodes, textStart + 1);
            boolean singleNode = textStart + 1 == textEnd;
            String text;
            if (singleNode) {
                text = ((HtmlText) nodes.get(textStart)).text();
            } else {
                StringBuilder sb = new StringBuilder();
                for (int i = textStart; i < textEnd; ++i) {
                    sb.append(((HtmlText) nodes.get(i)).text());
                }
                text = new String(sb);
                nodes.subList(textStart, textEnd - 1).clear();
            }
            List<UrlInfo> urlInfos = urlsExtractor.extractUrls(text);
            int size = urlInfos.size();
            if (size == 0) {
                if (!singleNode) {
                    nodes.set(textStart, new HtmlText(escaping, text));
                }
            } else {
                List<HtmlNode> newNodes = new ArrayList<>();
                int prev = 0;
                for (int i = 0; i < size; ++i) {
                    UrlInfo info = urlInfos.get(i);
                    FastUri uri = info.url();
                    int start = info.start();
                    int end = info.end();
                    if (start > prev) {
                        newNodes.add(
                            new HtmlText(
                                escaping,
                                text.substring(prev, start)));
                    }
                    prev = end;
                    BasicAttrs attrs = new BasicAttrs(1);
                    attrs.put(hrefAttr, new Attr(uri.toString(), hrefConfig));
                    HtmlTag a = new HtmlTag(
                        HtmlTextEscapingMode.PCDATA,
                        aTag,
                        Collections.emptyList(),
                        Collections.emptyList(),
                        null,
                        null,
                        attrs.compact(),
                        EmptyStyle.INSTANCE);
                    a.addTextNode(text.substring(start, end));
                    newNodes.add(a);
                }
                if (prev < text.length()) {
                    newNodes.add(new HtmlText(escaping, text.substring(prev)));
                }
                size = newNodes.size();
                if (size == 1) {
                    nodes.set(textStart, newNodes.get(0));
                } else {
                    nodes.remove(textStart);
                    nodes.addAll(textStart, newNodes);
                    --pos;
                    pos += size;
                }
            }
        }
    }

    private static int findTextNode(
        final List<HtmlNode> nodes,
        final int fromIndex)
    {
        int size = nodes.size();
        for (int i = fromIndex; i < size; ++i) {
            HtmlNode node = nodes.get(i);
            if (node instanceof HtmlText) {
                return i;
            }
        }
        return -1;
    }

    private static int findNonTextNode(
        final List<HtmlNode> nodes,
        final int fromIndex)
    {
        int size = nodes.size();
        for (int i = fromIndex; i < size; ++i) {
            HtmlNode node = nodes.get(i);
            if (!(node instanceof HtmlText)) {
                return i;
            }
        }
        return size;
    }
}

