package ru.yandex.sanitizer2;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.CharacterCodingException;
import java.util.Set;
import java.util.function.Predicate;

import com.twmacinta.util.MD5;

import ru.yandex.net.uri.fast.FastUri;
import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;
import ru.yandex.sanitizer2.config.ImmutableResizerConfig;
import ru.yandex.util.string.HexStrings;
import ru.yandex.util.string.StringUtils;

public class ResizerUrlRewriter extends AbstractUrlRewriter {
    private static final String URL = "url=";
    private static final int DEFAULT_MAX_CAPACITY = 1024;
    // PctEncoder with custom table taken from
    // arcadia/util/string/quote.cpp?rev=3937560#L195-200
    // which is called from
    // arcadia/yweb/mail/wmiarc/cacher/url_processor.h?rev=3937560#L104
    private static final ThreadLocal<PctEncoder.CompactEncoder> ENCODER =
        ThreadLocal.withInitial(
            () -> new PctEncoder.CompactEncoder(
                DEFAULT_MAX_CAPACITY,
                ResizerEncodingRule.INSTANCE));

    private final String salt;
    private final Predicate<String> whitelistHosts;
    private final String scheme;
    private final String userInfo;
    private final String host;
    private final int port;
    private final String path;
    private final String query;
    private final String fragment;
    private final String schemeSpecificPart;
    private final String authority;

    // CSOFF: ParameterNumber
    public ResizerUrlRewriter(
        final Set<String> bypassSchemes,
        final Set<String> allowedSchemes,
        final UrlRewriter next,
        final ImmutableResizerConfig resizerConfig)
    {
        super(bypassSchemes, allowedSchemes, next);
        salt = resizerConfig.salt();
        whitelistHosts = resizerConfig.whitelistHosts();
        URI uri = resizerConfig.uri();
        scheme = uri.getScheme();
        userInfo = uri.getRawUserInfo();
        host = uri.getHost();
        port = uri.getPort();
        path = uri.getRawPath();
        String query = uri.getRawQuery();
        if (query == null) {
            this.query = URL;
        } else {
            this.query = StringUtils.concat(query, '&', URL);
        }
        fragment = uri.getRawFragment();
        schemeSpecificPart = uri.getRawSchemeSpecificPart();
        authority = uri.getRawAuthority();
    }
    // CSON: ParameterNumber

    @Override
    public FastUri sanitize(final FastUri uri, final SanitizingContext context)
        throws URISyntaxException
    {
        FastUri tmp;
        String host = uri.host();
        if (host != null
            && !whitelistHosts.test(host)
            && ("http".equals(uri.scheme())
                || "https".equals(uri.scheme())))
        {
            try {
                StringBuilder uriSb =
                    new StringBuilder(uri.expectedStringLength());
                uri.toStringBuilder(uriSb);
                StringBuilder keySb = new StringBuilder(URL);
                int escapedUrlStart = keySb.length();
                ENCODER.get().process(uriSb, keySb);
                int escapedUrlEnd = keySb.length();
                keySb.append("/proxy=yes");
                keySb.append(salt);
                MD5 md5 = new MD5();
                md5.Update(StringUtils.getUtf8Bytes(new String(keySb)));
                StringBuilder sb = new StringBuilder(query);
                sb.append(keySb, escapedUrlStart, escapedUrlEnd);
                sb.append("&proxy=yes&key=");
                HexStrings.LOWER.toStringBuilder(
                    sb,
                    md5.Final());
                tmp = new FastUri(
                    scheme,
                    userInfo,
                    this.host,
                    port,
                    path,
                    new String(sb),
                    fragment,
                    schemeSpecificPart,
                    authority);
            } catch (CharacterCodingException e) {
                return null;
            }
        } else {
            tmp = uri;
        }
        return next.sanitize(tmp, context);
    }

    private static class ResizerEncodingRule extends PctEncodingRule {
        private static final ResizerEncodingRule INSTANCE =
            new ResizerEncodingRule();

        private final boolean[] encodingTable;

        ResizerEncodingRule() {
            encodingTable = new boolean[] {
                false, false, false, false, false, false, false, false,
                false, false, false, false, false, false, false, false,
                false, false, false, false, false, false, false, false,
                false, false, false, false, false, false, false, false,
                false, true, false, false, false, false, false, true,
                true, true, true, false, false, true, true, false,
                true, true, true, true, true, true, true, true,
                true, true, false, false, false, false, false, false,
                false, true, true, true, true, true, true, true,
                true, true, true, true, true, true, true, true,
                true, true, true, true, true, true, true, true,
                true, true, true, false, false, false, true, true,
                false, true, true, true, true, true, true, true,
                true, true, true, true, true, true, true, true,
                true, true, true, true, true, true, true, true,
                true, true, true, false, false, false, true, false
            };
        }

        @Override
        protected boolean[] getEncodingTable() {
            return encodingTable;
        }
    }
}

