package ru.yandex.sanitizer2;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.ExecutorServiceCloser;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericBiConsumer;
import ru.yandex.function.GenericNonThrowingCloseableAdapter;
import ru.yandex.function.HashMapFactory;
import ru.yandex.http.server.async.BaseAsyncServer;
import ru.yandex.http.server.async.DelegatedHttpAsyncRequestHandler;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.sanitizer2.config.ImmutableSanitizer2Config;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.FixedSetStater;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.stater.ThreadPoolStater;

public class Sanitizer2 extends BaseAsyncServer<ImmutableSanitizer2Config> {
    private final ThreadPoolExecutor threadPool;

    public Sanitizer2(final ImmutableSanitizer2Config config)
        throws IOException, PageHeaderException
    {
        super(config);
        threadPool = new ThreadPoolExecutor(
            config.workers(),
            config.workers(),
            1,
            TimeUnit.MINUTES,
            new LifoWaitBlockingQueue<>(config.connections()),
            new NamedThreadFactory(
                getThreadGroup(),
                getName() + "-W-",
                true));
        closeChain.add(new ExecutorServiceCloser(threadPool));
        TimeFrameQueue<String> pageHeaders =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        TimeFrameQueue<Object> phishingLinks =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        Registrar registrar = new Registrar(pageHeaders, phishingLinks);
        config.sanitizers().traverse(registrar);
        Set<String> pageHeadersNames = new HashSet<>();
        config.sanitizers().traverse(
            (pattern, sanitizingConfig) -> pageHeadersNames.addAll(
                sanitizingConfig.pageHeaders().keySet()));
        registerStater(
            new FixedSetStater<>(
                pageHeaders,
                HashMapFactory.instance(),
                pageHeadersNames,
                "page-header-",
                "page-headers",
                "page headers",
                null,
                null));
        registerStater(
            new PassiveStaterAdapter<>(
                phishingLinks,
                new NamedStatsAggregatorFactory<>(
                    "phishing-links_ammm",
                    CountAggregatorFactory.INSTANCE)));
        registerStater(new Sanitizer2Stater(threadPool));
        registerStater(new ThreadPoolStater(threadPool, "thread-pool-"));
    }

    @Override
    public void start() throws IOException {
        threadPool.prestartAllCoreThreads();
        super.start();
    }

    private static class Sanitizer2Stater implements Stater {
        private final ThreadPoolExecutor executor;

        Sanitizer2Stater(final ThreadPoolExecutor executor) {
            this.executor = executor;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            int synchronousTasks = executor.getQueue().size();
            statsConsumer.stat("sanitize-queue_axxx", synchronousTasks);
            statsConsumer.stat("sanitize-queue_ammx", synchronousTasks);
            statsConsumer.stat("sanitize-queue_ammm", synchronousTasks);

            int activeWorkers = executor.getActiveCount();
            statsConsumer.stat("active-workers_axxx", activeWorkers);
            statsConsumer.stat("active-workers_ammx", activeWorkers);
            statsConsumer.stat("active-workers_ammm", activeWorkers);
        }
    }

    private static class PhishingLinksCallback implements Runnable {
        private static final Object OBJECT = new Object();
        private final TimeFrameQueue<Object> phishingLinks;

        PhishingLinksCallback(final TimeFrameQueue<Object> phishingLinks) {
            this.phishingLinks = phishingLinks;
        }

        @Override
        public void run() {
            phishingLinks.accept(OBJECT);
        }
    }

    private class Registrar
        implements GenericBiConsumer<
            Pattern<RequestInfo>,
            ImmutableSanitizingConfig,
            PageHeaderException>
    {
        private final TimeFrameQueue<String> pageHeaders;
        private final Runnable phishingLinksCallback;

        Registrar(
            final TimeFrameQueue<String> pageHeaders,
            final TimeFrameQueue<Object> phishingLinks)
        {
            this.pageHeaders = pageHeaders;
            phishingLinksCallback = new PhishingLinksCallback(phishingLinks);
        }

        @Override
        public void accept(
            final Pattern<RequestInfo> pattern,
            final ImmutableSanitizingConfig config)
            throws PageHeaderException
        {
            SanitizingHandler handler =
                new SanitizingHandler(
                    config,
                    pageHeaders,
                    phishingLinksCallback);
            register(
                pattern,
                new DelegatedHttpAsyncRequestHandler<>(
                    handler,
                    Sanitizer2.this,
                    threadPool),
                RequestHandlerMapper.POST);
            closeChain.add(new GenericNonThrowingCloseableAdapter<>(handler));
        }
    }
}

