package ru.yandex.sanitizer2;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Comparator;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;

import com.helger.css.decl.CSSSelector;
import com.helger.css.decl.CSSStyleRule;

import ru.yandex.function.StringBuilderable;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.util.string.StringUtils;

public class SimpleStyleProcessor implements StyleProcessor {
    private static final int NODE_MAX_SELECTORS = 8;
    private static final int TAG_FLAG = 1;
    private static final int CLASS_FLAG = 2;
    private static final int ID_FLAG = 4;
    private static final String ASTERISK = "*";

    // Maps selector to properties list. Selector can be:
    // *, tag, .class, tag.class, #id, tag#id, #id.class, tag#id.class
    // For specifity ordering see:
    // https://gist.github.com/mjj2000/5873872
    // http://www.w3.org/TR/CSS2/cascade.html#specificity
    private final Map<String, List<CssRule>> css = new HashMap<>();
    // This map contains only tag rules, as tag names are interned
    private final IdentityHashMap<String, List<CssRule>> tagsCss =
        new IdentityHashMap<>();
    // Temporary buffer for matched css rules
    private final List<CssRule> matchedRules =
        new ArrayList<>(NODE_MAX_SELECTORS);
    private final BasicStyle tmpStyle = new BasicStyle();
    private final List<CssRule> tmpRules = new ArrayList<>(2);
    private final BitSet selectorsTypes =
        new BitSet(TAG_FLAG + CLASS_FLAG + ID_FLAG + 1);
    private final ImmutableSanitizingConfig config;
    private int rulesCounter = 0;

    public SimpleStyleProcessor(final ImmutableSanitizingConfig config) {
        this.config = config;
    }

    private void matchRules(final String selector) {
        List<CssRule> rules = css.get(selector);
        if (rules != null) {
            matchedRules.addAll(rules);
        }
    }

    private void matchTagRules(final String selector) {
        List<CssRule> rules = tagsCss.get(selector);
        if (rules != null) {
            matchedRules.addAll(rules);
        }
    }

    @Override
    public void pushTag(@Nonnull final AbstractHtmlTag tag) {
    }

    @Override
    public void applyStyle(@Nonnull final AbstractHtmlTag tag) {
        String id = tag.id();
        if (id != null) {
            id = StringUtils.concat('#', id);
        }

        matchedRules.clear();
        if (selectorsTypes.get(0)) {
            matchTagRules(ASTERISK);
        }
        String tagName = tag.tagName();
        if (selectorsTypes.get(TAG_FLAG)) {
            matchTagRules(tagName);
        }
        List<String> classes = tag.classes();
        int classesCount = classes.size();
        if (classesCount != 0) {
            if (selectorsTypes.get(CLASS_FLAG)) {
                for (int i = 0; i < classesCount; ++i) {
                    matchRules(StringUtils.concat('.', classes.get(i)));
                }
            }
            if (selectorsTypes.get(TAG_FLAG | CLASS_FLAG)) {
                for (int i = 0; i < classesCount; ++i) {
                    matchRules(
                        StringUtils.concat(tagName, '.', classes.get(i)));
                }
            }
        }
        if (id != null) {
            if (selectorsTypes.get(ID_FLAG)) {
                matchRules(id);
            }
            String tagId = StringUtils.concat(tagName, id);
            if (selectorsTypes.get(TAG_FLAG | ID_FLAG)) {
                matchRules(tagId);
            }
            if (classesCount != 0) {
                if (selectorsTypes.get(ID_FLAG | CLASS_FLAG)) {
                    for (int i = 0; i < classesCount; ++i) {
                        matchRules(
                            StringUtils.concat(id, '.', classes.get(i)));
                    }
                }
                if (selectorsTypes.get(TAG_FLAG | ID_FLAG | CLASS_FLAG)) {
                    for (int i = 0; i < classesCount; ++i) {
                        matchRules(
                            StringUtils.concat(tagId, '.', classes.get(i)));
                    }
                }
            }
        }

        int matchedRulesSize = matchedRules.size();
        if (matchedRulesSize > 0) {
            matchedRules.sort(CssRuleComparator.INSTANCE);
            tmpStyle.clear();
            for (int i = 0; i < matchedRulesSize; ++i) {
                tmpStyle.merge(matchedRules.get(i).style);
            }
            tmpStyle.merge(tag.style());
            Style calculatedStyle = tmpStyle.compact();
            if (calculatedStyle == tmpStyle) {
                // Not compacted
                // Copy tmpStyle, so it won't be externally modified
                calculatedStyle = new BasicStyle(calculatedStyle);
            }
            tag.style(calculatedStyle);
        }
    }

    @Override
    public void popTag() {
    }

    public boolean processStyleRule(
        final CSSSelector selector,
        final List<CssDeclaration> declarations,
        final SanitizingContext context)
    {
        int membersCount = selector.getMemberCount();
        String tagName = null;
        String className = null;
        String idName = null;
        for (int j = 0; j < membersCount; ++j) {
            String member =
                selector.getMemberAtIndex(j).getAsCSSString();
            if (!member.isEmpty()) {
                char c = member.charAt(0);
                if (j == 0
                    && (c >= 'A' && (c <= 'Z' || (c <= 'z' && c >= 'a'))))
                {
                    // this is a tag selector, lowercase it
                    tagName = member.toLowerCase(Locale.ROOT);
                    String internedTagName = config.internTag(tagName);
                    if (internedTagName != null) {
                        tagName = internedTagName;
                    }
                } else if (c == '.') {
                    if (className == null) {
                        // class selector
                        className = member.substring(1);
                    } else {
                        return false;
                    }
                } else if (c == '#') {
                    if (idName == null) {
                        // id/anchor name selector
                        idName = member.substring(1);
                    } else {
                        return false;
                    }
                } else if (c != '*' || member.length() != 1) {
                    return false;
                }
            }
        }
        tmpRules.add(
            new CssRule(
                tagName,
                className,
                idName,
                rulesCounter++,
                declarations));
        return true;
    }

    @Override
    public boolean processStyleRule(
        @Nonnull final CSSStyleRule styleRule,
        @Nonnull final SanitizingContext context)
    {
        List<CssDeclaration> declarations =
            new ArrayList<>(styleRule.getDeclarationCount());
        BasicStyle.convertDeclarations(
            declarations,
            config,
            styleRule,
            context);
        tmpRules.clear();
        int selectorsCount = styleRule.getSelectorCount();
        for (int i = 0; i < selectorsCount; ++i) {
            CSSSelector selector = styleRule.getSelectorAtIndex(i);
            if (!processStyleRule(selector, declarations, context)
                && config.complexCssMatching())
            {
                return false;
            }
        }
        selectorsCount = tmpRules.size();
        for (int i = 0; i < selectorsCount; ++i) {
            CssRule rule = tmpRules.get(i);
            selectorsTypes.set(rule.flags);
            if (rule.flags <= TAG_FLAG) {
                tagsCss.computeIfAbsent(
                    rule.selectorString,
                    ArrayListFactory.INSTANCE)
                    .add(rule);
            } else {
                css.computeIfAbsent(
                    rule.selectorString,
                    ArrayListFactory.INSTANCE)
                    .add(rule);
            }
        }
        return true;
    }

    @Override
    public StyleProcessor upgrade(final ImmutableSanitizingConfig config) {
        ComplexStyleProcessor styleProcessor =
            new ComplexStyleProcessor(config);
        if (!css.isEmpty()) {
            for (Map.Entry<String, List<CssRule>> entry: css.entrySet()) {
                for (CssRule rule: entry.getValue()) {
                    styleProcessor.processSimpleStyleRule(
                        rule.tagName,
                        rule.className,
                        rule.idName,
                        rule.style);
                }
            }
        }
        if (!tagsCss.isEmpty()) {
            for (Map.Entry<String, List<CssRule>> entry: tagsCss.entrySet()) {
                for (CssRule rule: entry.getValue()) {
                    styleProcessor.processSimpleStyleRule(
                        rule.tagName,
                        null,
                        null,
                        rule.style);
                }
            }
        }
        return styleProcessor;
    }

    private static class CssRule {
        private final String tagName;
        private final String className;
        private final String idName;
        private final int id;
        private final List<CssDeclaration> style;
        private final String selectorString;
        private final int flags;

        // CSOFF: ParameterNumber
        CssRule(
            final String tagName,
            final String className,
            final String idName,
            final int id,
            final List<CssDeclaration> style)
        {
            this.tagName = tagName;
            this.className = className;
            this.idName = idName;
            this.id = id;
            this.style = style;
            int flags = 0;
            StringBuilder sb = new StringBuilder(
                StringBuilderable.calcExpectedStringLength(tagName, 0)
                + StringBuilderable.calcExpectedStringLength(className, 0)
                + StringBuilderable.calcExpectedStringLength(idName, 0)
                + 2);
            if (tagName != null) {
                flags = TAG_FLAG;
                sb.append(tagName);
            }
            if (idName != null) {
                flags |= ID_FLAG;
                sb.append('#');
                sb.append(idName);
            }
            // Order is changed here, because processTag(...) adds classes at
            // the final step
            if (className != null) {
                flags |= CLASS_FLAG;
                sb.append('.');
                sb.append(className);
            }
            this.flags = flags;
            if (sb.length() == 0) {
                selectorString = ASTERISK;
            } else if (flags == TAG_FLAG) {
                selectorString = tagName;
            } else {
                selectorString = new String(sb);
            }
        }
        // CSON: ParameterNumber
    }

    private enum CssRuleComparator implements Comparator<CssRule> {
        INSTANCE;

        @Override
        public int compare(final CssRule lhs, final CssRule rhs) {
            int cmp = Integer.compare(lhs.flags, rhs.flags);
            if (cmp == 0) {
                cmp = Integer.compare(lhs.id, rhs.id);
            }
            return cmp;
        }
    }

    private enum ArrayListFactory implements Function<String, List<CssRule>> {
        INSTANCE;

        @Override
        public List<CssRule> apply(final String selector) {
            return new ArrayList<>(2);
        }
    }
}

