package ru.yandex.sanitizer2;

import java.util.Arrays;

import javax.annotation.Nonnull;

import com.helger.css.ECSSVersion;
import com.helger.css.decl.CSSFontFaceRule;
import com.helger.css.decl.CSSKeyframesRule;
import com.helger.css.decl.CSSMediaRule;
import com.helger.css.decl.CSSPageRule;
import com.helger.css.decl.CSSStyleRule;
import com.helger.css.decl.CSSSupportsRule;
import com.helger.css.decl.CSSViewportRule;
import com.helger.css.decl.CascadingStyleSheet;
import com.helger.css.decl.visit.CSSVisitor;
import com.helger.css.decl.visit.DefaultCSSVisitor;
import com.helger.css.handler.CSSHandler;
import com.helger.css.parser.ParseException;
import com.helger.css.parser.ParserCSS30;
import com.helger.css.parser.ParserCSS30TokenManager;
import com.helger.css.reader.errorhandler.DoNothingCSSInterpretErrorHandler;
import com.helger.css.reader.errorhandler.DoNothingCSSParseErrorHandler;
import com.helger.css.reader.errorhandler.ICSSInterpretErrorHandler;
import com.helger.css.reader.errorhandler.ICSSParseErrorHandler;

import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;

public class StyleExtractor extends DefaultCSSVisitor {
    private static final ICSSParseErrorHandler PARSE_ERROR_HANDLER =
        new DoNothingCSSParseErrorHandler();
    private static final ICSSInterpretErrorHandler INTERPRETER_ERROR_HANDLER =
        new DoNothingCSSInterpretErrorHandler();
    private static final boolean[] EMPTY_RULES_STACK = new boolean[0];

    private final ImmutableSanitizingConfig config;
    private final SanitizingContext context;
    private StyleProcessor styleProcessor = EmptyStyleProcessor.INSTANCE;
    private boolean[] rulesStack = EMPTY_RULES_STACK;
    private int rulesStackSize = 0;

    public StyleExtractor(
        @Nonnull final ImmutableSanitizingConfig config,
        @Nonnull final SanitizingContext context)
    {
        this.config = config;
        this.context = context;
    }

    public StyleProcessor styleProcessor() {
        return styleProcessor;
    }

    private static CascadingStyleSheet parseCss(
        @Nonnull final StringBuilder text)
        throws ParseException
    {
        ParserCSS30TokenManager tokenManager =
            new ParserCSS30TokenManager(new StringBuilderCharStream(text));
        tokenManager.setCustomErrorHandler(PARSE_ERROR_HANDLER);
        ParserCSS30 parser = new ParserCSS30(tokenManager);
        parser.setBrowserCompliantMode(true);
        parser.setCustomErrorHandler(PARSE_ERROR_HANDLER);
        return CSSHandler.readCascadingStyleSheetFromNode(
            ECSSVersion.CSS30,
            INTERPRETER_ERROR_HANDLER,
            false,
            parser.styleSheet());
    }

    private static boolean isAcceptableMedia(@Nonnull final String media) {
        switch (media) {
            case "all":
            case "screen":
            case "only screen":
                return true;
            default:
                return false;
        }
    }

    public void parseStyleBlock(@Nonnull final HtmlCDataTag cdata) {
        boolean acceptableMedia;
        Attr media = cdata.attrs().get("media");
        if (media == null) {
            acceptableMedia = true;
        } else {
            acceptableMedia = isAcceptableMedia(media.value());
        }

        boolean acceptableType;
        if (acceptableMedia) {
            Attr type = cdata.attrs().get("type");
            if (type == null) {
                acceptableType = true;
            } else {
                acceptableType = "text/css".equals(type.value());
            }
        } else {
            acceptableType = false;
        }

        boolean acceptableStyle;
        if (acceptableType) {
            acceptableStyle =
                !cdata.attrs().containsKey("amp4email-boilerplate");
        } else {
            acceptableStyle = false;
        }

        if (acceptableStyle) {
            CascadingStyleSheet css;
            try {
                css = parseCss(cdata.text());
            } catch (RuntimeException | ParseException e) {
                css = null;
            }
            if (css != null) {
                CSSVisitor.visitCSS(css, this);
            }
        }
    }

    private void addRuleToStack(final boolean b) {
        if (rulesStackSize >= rulesStack.length) {
            rulesStack = Arrays.copyOf(rulesStack, (rulesStackSize + 1) << 1);
        }
        rulesStack[rulesStackSize++] = b;
    }

    private boolean allowStyleRule() {
        for (int i = 0; i < rulesStackSize; ++i) {
            if (!rulesStack[i]) {
                return false;
            }
        }
        return true;
    }

    @Override
    public void onBeginStyleRule(final CSSStyleRule styleRule) {
        if (allowStyleRule()) {
            while (true) {
                if (styleProcessor.processStyleRule(styleRule, context)) {
                    break;
                } else {
                    styleProcessor = styleProcessor.upgrade(config);
                }
            }
        }
    }

    @Override
    public void onBeginPageRule(final CSSPageRule pageRule) {
        addRuleToStack(false);
    }

    @Override
    public void onEndPageRule(final CSSPageRule pageRule) {
        --rulesStackSize;
    }

    @Override
    public void onBeginFontFaceRule(final CSSFontFaceRule fontFaceRule) {
        addRuleToStack(false);
    }

    @Override
    public void onEndFontFaceRule(final CSSFontFaceRule montFaceRule) {
        --rulesStackSize;
    }

    @Override
    public void onBeginMediaRule(final CSSMediaRule mediaRule) {
        int count = mediaRule.getMediaQueryCount();
        boolean acceptable =
            count == 0
            || (count == 1
                && isAcceptableMedia(
                    mediaRule.getMediaQueryAtIndex(0).getAsCSSString()));
        addRuleToStack(acceptable);
    }

    @Override
    public void onEndMediaRule(final CSSMediaRule mediaRule) {
        --rulesStackSize;
    }

    @Override
    public void onBeginKeyframesRule(final CSSKeyframesRule keyframesRule) {
        addRuleToStack(false);
    }

    @Override
    public void onEndKeyframesRule(final CSSKeyframesRule keyframesRule) {
        --rulesStackSize;
    }

    @Override
    public void onBeginViewportRule(final CSSViewportRule viewportRule) {
        addRuleToStack(false);
    }

    @Override
    public void onEndViewportRule(final CSSViewportRule viewportRule) {
        --rulesStackSize;
    }

    @Override
    public void onBeginSupportsRule(final CSSSupportsRule supportsRule) {
        addRuleToStack(false);
    }

    @Override
    public void onEndSupportsRule(final CSSSupportsRule supportsRule) {
        --rulesStackSize;
    }
}

