package ru.yandex.sanitizer2;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;

import org.owasp.html.HtmlTextEscapingMode;

import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;

public class TextPlainProcessor implements Function<String, HtmlTag> {
    private static final int INITIAL_STACK_CAPACITY = 8;

    private final HtmlTag root = new HtmlTag(
        HtmlTextEscapingMode.PCDATA,
        "",
        Collections.emptyList(),
        Collections.emptyList(),
        null,
        null,
        EmptyAttrs.INSTANCE,
        EmptyStyle.INSTANCE);
    // temporary string builder for line conversion
    private final StringBuilder sb = new StringBuilder();
    private final String brTag;
    private final String blockquoteTag;
    private final String pTag;
    private final String spanTag;
    private HtmlTag[] stack = new HtmlTag[INITIAL_STACK_CAPACITY];
    private int stackSize = 1;

    public TextPlainProcessor(final ImmutableSanitizingConfig config) {
        brTag = config.brTag();
        blockquoteTag = config.blockquoteTag();
        pTag = config.pTag();
        spanTag = config.spanTag();
        stack[0] = root;
    }

    @Override
    public HtmlTag apply(final String text) {
        String[] lines = text.split("\n");
        State state = State.INITIAL;
        int currentQuotationLevel = 0;
        for (String line: lines) {
            String normalizedLine = line.replace("\r", "");
            int lineQuotationLevel = quotationLevel(normalizedLine);
            Transition transition = state.process(
                normalizedLine.substring(lineQuotationLevel),
                currentQuotationLevel,
                lineQuotationLevel);
            if (state != transition.state
                || currentQuotationLevel != lineQuotationLevel)
            {
                state.onLeave(
                    this,
                    currentQuotationLevel,
                    lineQuotationLevel);
                transition.state.onEnter(
                    this,
                    currentQuotationLevel,
                    lineQuotationLevel);
            }
            state = transition.state;
            addLine(stack[stackSize - 1], transition.line);
            currentQuotationLevel = lineQuotationLevel;
        }
        state.onLeave(this, currentQuotationLevel, 0);
        root.accept(new PgpSignatureFoldingVisitor(blockquoteTag));
        return root;
    }

    private static int quotationLevel(final String line) {
        int level = 0;
        int length = line.length();
        while (level < length && line.charAt(level) == '>') {
            ++level;
        }
        return level;
    }

    private String prepareLine(final String line) {
        sb.setLength(0);
        int pos = 0;
        int length = line.length();
        while (pos < length) {
            char c = line.charAt(pos);
            if (c == ' ') {
                sb.append('\u00a0');
            } else if (c == '\t') {
                sb.append(
                    "\u00a0\u00a0\u00a0\u00a0\u00a0\u00a0\u00a0\u00a0");
            } else {
                break;
            }
            ++pos;
        }
        if (pos > 0) {
            sb.append(line.substring(pos));
            return sb.toString();
        } else {
            return line;
        }
    }

    private void addLine(final HtmlTag tag, final String line) {
        tag.addTextNode(prepareLine(line));
        tag.addNode(
            new HtmlTag(
                HtmlTextEscapingMode.VOID,
                brTag,
                Collections.emptyList(),
                Collections.emptyList(),
                null,
                null,
                EmptyAttrs.INSTANCE,
                EmptyStyle.INSTANCE));
    }

    private HtmlTag lastTag() {
        return stack[stackSize - 1];
    }

    private void addTag(final HtmlTag tag) {
        if (stackSize == stack.length) {
            stack = Arrays.copyOf(stack, stack.length << 1);
        }
        stack[stackSize++] = tag;
    }

    private void ensureStackCapacity(final int newCapacity) {
        if (newCapacity > stack.length) {
            stack = Arrays.copyOf(
                stack,
                Math.max(newCapacity, stack.length << 1));
        }
    }

    private enum LineType {
        NORMAL,
        SIGNATURE_START;

        public static LineType detectLineType(final String line) {
            if (line.equals("-- ") || line.equals("--")) {
                return SIGNATURE_START;
            } else {
                return NORMAL;
            }
        }
    }

    private enum State {
        INITIAL {
            // currentQuotationLevel expected to be zero
            @Override
            public Transition process(
                final String line,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                if (lineQuotationLevel == 0) {
                    LineType lineType = LineType.detectLineType(line);
                    if (lineType == LineType.SIGNATURE_START) {
                        return new Transition("-- ", State.IN_SIGNATURE);
                    } else {
                        return new Transition(line, State.IN_PARAGRAPH);
                    }
                } else {
                    return new Transition(line, State.IN_BLOCKQUOTE);
                }
            }

            @Override
            public void onEnter(
                final TextPlainProcessor processor,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
            }

            @Override
            public void onLeave(
                final TextPlainProcessor processor,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
            }
        },
        IN_PARAGRAPH {
            // currentQuotationLevel expected to be zero
            @Override
            public Transition process(
                final String line,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                if (lineQuotationLevel == 0) {
                    LineType lineType = LineType.detectLineType(line);
                    if (lineType == LineType.SIGNATURE_START) {
                        return new Transition("-- ", State.IN_SIGNATURE);
                    } else {
                        return new Transition(line, State.IN_PARAGRAPH);
                    }
                } else {
                    return new Transition(line, State.IN_BLOCKQUOTE);
                }
            }

            @Override
            public void onEnter(
                final TextPlainProcessor processor,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                HtmlTag node = processor.lastTag();
                HtmlTag paragraphNode = new HtmlTag(
                    HtmlTextEscapingMode.PCDATA,
                    processor.pTag,
                    Collections.emptyList(),
                    Collections.emptyList(),
                    null,
                    null,
                    EmptyAttrs.INSTANCE,
                    EmptyStyle.INSTANCE);
                node.addNode(paragraphNode);
                processor.addTag(paragraphNode);
            }

            @Override
            public void onLeave(
                final TextPlainProcessor processor,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                --processor.stackSize;
            }
        },
        IN_SIGNATURE {
            // currentQuotationLevel expected to be zero
            @Override
            public Transition process(
                final String line,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                if (lineQuotationLevel == 0) {
                    return new Transition(line, State.IN_SIGNATURE);
                } else {
                    return new Transition(line, State.IN_BLOCKQUOTE);
                }
            }

            @Override
            public void onEnter(
                final TextPlainProcessor processor,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                HtmlTag node = processor.lastTag();
                List<String> classes = Collections.singletonList("wmi-sign");
                HtmlTag signatureNode = new HtmlTag(
                    HtmlTextEscapingMode.PCDATA,
                    processor.spanTag,
                    classes,
                    classes,
                    null,
                    null,
                    EmptyAttrs.INSTANCE,
                    EmptyStyle.INSTANCE);
                node.addNode(signatureNode);
                processor.addTag(signatureNode);
            }

            @Override
            public void onLeave(
                final TextPlainProcessor processor,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                --processor.stackSize;
            }
        },
        IN_BLOCKQUOTE {
            // currentQuotationLevel expected to be non zero
            @Override
            public Transition process(
                final String line,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                if (currentQuotationLevel == lineQuotationLevel) {
                    return new Transition(line, State.IN_BLOCKQUOTE);
                } else if (currentQuotationLevel < lineQuotationLevel) {
                    return new Transition(line, State.IN_BLOCKQUOTE);
                } else {
                    if (lineQuotationLevel == 0) {
                        return State.INITIAL.process(line, 0, 0);
                    } else {
                        return new Transition(line, State.IN_BLOCKQUOTE);
                    }
                }
            }

            @Override
            public void onEnter(
                final TextPlainProcessor processor,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                List<String> classes = Collections.singletonList("wmi-quote");
                processor.ensureStackCapacity(
                    processor.stackSize
                    + lineQuotationLevel - currentQuotationLevel);
                HtmlTag node = processor.lastTag();
                for (int i = currentQuotationLevel; i < lineQuotationLevel; ++i) {
                    HtmlTag blockquotedNode = new HtmlTag(
                        HtmlTextEscapingMode.PCDATA,
                        processor.blockquoteTag,
                        classes,
                        classes,
                        null,
                        null,
                        EmptyAttrs.INSTANCE,
                        EmptyStyle.INSTANCE);
                    node.addNode(blockquotedNode);
                    processor.stack[processor.stackSize++] = blockquotedNode;
                    node = blockquotedNode;
                }
            }

            @Override
            public void onLeave(
                final TextPlainProcessor processor,
                final int currentQuotationLevel,
                final int lineQuotationLevel)
            {
                if (currentQuotationLevel > lineQuotationLevel) {
                    processor.stackSize -=
                        currentQuotationLevel - lineQuotationLevel;
                }
            }
        };

        // Line will be already normalized and quotation stripped
        public abstract Transition process(
            String line,
            int currentQuotationLevel,
            int lineQuotationLevel);

        public abstract void onEnter(
            TextPlainProcessor processor,
            int currentQuotationLevel,
            int lineQuotationLevel);

        public abstract void onLeave(
            TextPlainProcessor processor,
            int currentQuotationLevel,
            int lineQuotationLevel);
    }

    private static class Transition {
        private final String line;
        private final State state;

        Transition(final String line, final State state) {
            this.line = line;
            this.state = state;
        }
    }
}

