package ru.yandex.sanitizer2;

import java.net.URISyntaxException;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.net.uri.fast.FastUri;
import ru.yandex.net.uri.fast.FastUriParser;
import ru.yandex.sanitizer2.config.ImmutableHiderefererConfig;
import ru.yandex.sanitizer2.config.ImmutableResizerConfig;
import ru.yandex.sanitizer2.config.ImmutableUrlConfig;
import ru.yandex.sanitizer2.config.ImmutableUrlSanitizingConfig;
import ru.yandex.util.string.StringUtils;

public interface UrlRewriter extends AttributeSanitizer {
    String HTTP_SHORT_SCHEME_PREFIX = "http:/";
    String HTTP_LONG_SCHEME_PREFIX = "http:///";
    int HTTP_SHORT_SCHEME_PREFIX_LENGTH = HTTP_SHORT_SCHEME_PREFIX.length();
    int HTTP_SCHEME_PREFIX_LENGTH = HTTP_SHORT_SCHEME_PREFIX_LENGTH + 1;
    String HTTPS_SHORT_SCHEME_PREFIX = "https:/";
    String HTTPS_LONG_SCHEME_PREFIX = "https:///";
    int HTTPS_SHORT_SCHEME_PREFIX_LENGTH = HTTPS_SHORT_SCHEME_PREFIX.length();
    int HTTPS_SCHEME_PREFIX_LENGTH = HTTPS_SHORT_SCHEME_PREFIX_LENGTH + 1;

    // Scheme expected to be lowercased or null
    boolean bypassScheme(final String scheme);

    boolean allowScheme(final String scheme);

    FastUri sanitize(FastUri uri, SanitizingContext context)
        throws URISyntaxException;

    @Override
    @Nullable
    default String sanitize(
        @Nonnull final String value,
        @Nonnull final SanitizingContext context)
    {
        String result = null;
        try {
            FastUri uri = fixUri(value).parse();
            String uriScheme = StringUtils.nullifyEmpty(uri.scheme());
            String scheme = StringUtils.toLowerCaseOrNull(uriScheme);
            String uriHost = StringUtils.nullifyEmpty(uri.host());
            String host = StringUtils.toLowerCaseOrNull(uriHost);
            boolean bypassScheme = bypassScheme(scheme);
            if (!bypassScheme) {
                uri.host(host);
                uri.scheme(scheme);
            }

            if (scheme == null || allowScheme(scheme)) {
                if (scheme != null
                    || host != null
                    || uri.port() != -1
                    || StringUtils.nullifyEmpty(uri.userInfo()) != null
                    || StringUtils.nullifyEmpty(uri.path()) != null
                    || StringUtils.nullifyEmpty(uri.query()) != null
                    || StringUtils.nullifyEmpty(uri.fragment()) != null)
                {
                    uri = sanitize(uri, context);
                }
            } else if (bypassScheme) {
                // TODO: Make this configurable
                if ("data".equals(scheme)) {
                    String ssp = uri.schemeSpecificPart();
                    if (ssp == null) {
                        ssp = "";
                    }
                    int pos = 0;
                    while (pos < ssp.length() && ssp.charAt(pos) == '/') {
                        ++pos;
                    }
                    ssp = ssp.substring(pos);
                    if (!StringUtils.startsWithIgnoreCase(ssp, "image/")) {
                        uri = null;
                    }
                }
            } else {
                uri = null;
            }
            if (uri != null) {
                result = uri.toString();
            }
        } catch (URISyntaxException e) {
            // ignore and return null
        }
        return result;
    }

    static FastUriParser fixUri(final String value) {
        String tmp = value;
        if (tmp.startsWith(HTTP_SHORT_SCHEME_PREFIX)) {
            if (tmp.length() > HTTP_SHORT_SCHEME_PREFIX_LENGTH) {
                if (tmp.charAt(HTTP_SHORT_SCHEME_PREFIX_LENGTH) != '/') {
                    tmp = StringUtils.concat(
                        HTTP_SHORT_SCHEME_PREFIX,
                        '/',
                        tmp.substring(HTTP_SHORT_SCHEME_PREFIX_LENGTH));
                } else {
                    while (tmp.startsWith(HTTP_LONG_SCHEME_PREFIX)) {
                        tmp = StringUtils.concat(
                            HTTP_SHORT_SCHEME_PREFIX,
                            tmp.substring(HTTP_SCHEME_PREFIX_LENGTH));
                    }
                }
            }
        } else if (tmp.startsWith(HTTPS_SHORT_SCHEME_PREFIX)) {
            if (tmp.length() > HTTPS_SHORT_SCHEME_PREFIX_LENGTH) {
                if (tmp.charAt(HTTPS_SHORT_SCHEME_PREFIX_LENGTH) != '/') {
                    tmp = StringUtils.concat(
                        HTTPS_SHORT_SCHEME_PREFIX,
                        '/',
                        tmp.substring(HTTPS_SHORT_SCHEME_PREFIX_LENGTH));
                } else {
                    while (tmp.startsWith(HTTPS_LONG_SCHEME_PREFIX)) {
                        tmp = StringUtils.concat(
                            HTTPS_SHORT_SCHEME_PREFIX,
                            tmp.substring(HTTPS_SCHEME_PREFIX_LENGTH));
                    }
                }
            }
        }
        char[] input = tmp.toCharArray();
        for (int i = 0; i < input.length; ++i) {
            if (input[i] < ' ') {
                int skipped = 1;
                for (int j = i + 1; j < input.length; ++j) {
                    char c = input[j];
                    if (c >= ' ') {
                        input[j - skipped] = input[j];
                    } else {
                        ++skipped;
                    }
                }
                return new FastUriParser(input, 0, input.length - skipped);
            }
        }
        return new FastUriParser(input);
    }

    static String selectScheme(final boolean forceHttps) {
        if (forceHttps) {
            return "https";
        } else {
            return "http";
        }
    }

    static UrlRewriter create(
        final ImmutableUrlSanitizingConfig urlSanitizingConfig,
        final ImmutableUrlConfig urlConfig)
    {
        UrlRewriter urlRewriter = IdentityUrlRewriter.INSTANCE;
        Set<String> bypassSchemes = urlSanitizingConfig.bypassSchemes();
        Set<String> allowedSchemes = urlSanitizingConfig.allowedSchemes();
        ImmutableHiderefererConfig hiderefererConfig =
            urlSanitizingConfig.hiderefererConfig();
        if (hiderefererConfig != null) {
            urlRewriter = new HiderefererUrlRewriter(
                bypassSchemes,
                allowedSchemes,
                urlRewriter,
                hiderefererConfig);
        }
        if (urlSanitizingConfig.unproxy()) {
            urlRewriter = new SanitizingUrlRewriter(
                bypassSchemes,
                allowedSchemes,
                urlRewriter,
                urlConfig);
            if (!urlSanitizingConfig.allowRelativeUrls()) {
                urlRewriter = new UnrelativizingUrlRewriter(
                    bypassSchemes,
                    allowedSchemes,
                    urlRewriter,
                    selectScheme(urlSanitizingConfig.forceHttps()));
            }
            urlRewriter = new BaseHrefUrlRewriter(
                bypassSchemes,
                allowedSchemes,
                urlRewriter);
            urlRewriter = new UnproxyUrlRewriter(
                bypassSchemes,
                allowedSchemes,
                urlRewriter,
                urlSanitizingConfig.resizerConfig().uri());
        } else {
            ImmutableResizerConfig resizerConfig =
                urlSanitizingConfig.resizerConfig();
            if (resizerConfig == null || !urlConfig.image()) {
                urlRewriter = new SanitizingUrlRewriter(
                    bypassSchemes,
                    allowedSchemes,
                    urlRewriter,
                    urlConfig);
            } else {
                if (urlSanitizingConfig.forceHttps()) {
                    urlRewriter = new ForceHttpsUrlRewriter(
                        bypassSchemes,
                        allowedSchemes,
                        urlRewriter);
                }
                urlRewriter = new ResizerUrlRewriter(
                    bypassSchemes,
                    allowedSchemes,
                    urlRewriter,
                    resizerConfig);
                urlRewriter = new SanitizingUrlRewriter(
                    bypassSchemes,
                    allowedSchemes,
                    urlRewriter,
                    urlConfig);
            }
            if (!urlSanitizingConfig.allowRelativeUrls()) {
                urlRewriter = new UnrelativizingUrlRewriter(
                    bypassSchemes,
                    allowedSchemes,
                    urlRewriter,
                    selectScheme(urlSanitizingConfig.forceHttps()));
            }
            urlRewriter = new BaseHrefUrlRewriter(
                bypassSchemes,
                allowedSchemes,
                urlRewriter);
        }
        return urlRewriter;
    }
}

