package ru.yandex.sanitizer2.config;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;

import ru.yandex.function.GenericFunction;
import ru.yandex.parser.config.ConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.FiniteNumberValidator;

public abstract class AbstractPropertyConfigBuilder
    <T extends AbstractPropertyConfigBuilder<T>>
    implements ConfigBuilder<T>, PropertyConfig
{
    private static final GenericFunction<String, Double, Exception> PARSER =
        FiniteNumberValidator.<Double, Exception>instance()
            .compose(Double::valueOf);
    private static final String PX = "px";
    private static final double PX_IN_INCH = 96d;
    private static final double CM_IN_INCH = 2.54d;
    private static final double MM_IN_CM = 10d;
    private static final double PT_IN_INCH = 72d;
    private static final double PT_IN_PC = 12d;
    private static final double PX_IN_EM = 16d;
    private static final double PCT_IN_EM = 100d;
    // exotic units, not precise
    private static final double PX_IN_EX = 2.75391d;
    private static final double PX_IN_CH = 3d;
    private static final double PX_IN_REM = 16d;

    private UrlConfigBuilder urlConfig;
    private boolean inherited;
    private boolean trim;
    private boolean lowercase;
    private Pattern splitPattern;
    private String separator;
    private Set<String> preservedValues;
    private Function<String, String> obfuscator;
    private Pattern pattern;
    private Map<String, String> replacement;
    private Map<String, Double> minValue;
    private Map<String, Double> maxValue;
    private Pattern subpattern;
    private Map<String, String> subreplacement;

    protected AbstractPropertyConfigBuilder(final PropertyConfig config) {
        urlConfig(config.urlConfig());
        inherited(config.inherited());
        trim(config.trim());
        lowercase(config.lowercase());
        splitPattern(config.splitPattern());
        separator(config.separator());
        preservedValues(config.preservedValues());
        obfuscator(config.obfuscator());
        pattern(config.pattern());
        replacement(config.replacement());
        minValue(config.minValue());
        maxValue(config.maxValue());
        subpattern(config.subpattern());
        subreplacement(config.subreplacement());
    }

    protected AbstractPropertyConfigBuilder(
        final IniConfig config,
        final PropertyConfig defaults)
        throws ConfigException
    {
        IniConfig section = config.sectionOrNull("url");
        if (section == null) {
            urlConfig = null;
        } else {
            UrlConfig urlConfigDefaults = defaults.urlConfig();
            if (urlConfigDefaults == null) {
                urlConfig = new UrlConfigBuilder(section);
            } else {
                urlConfig = new UrlConfigBuilder(section, urlConfigDefaults);
            }
        }
        inherited = INHERITED.extract(config, defaults.inherited());
        trim = TRIM.extract(config, defaults.trim());
        lowercase = LOWERCASE.extract(config, defaults.lowercase());
        splitPattern = SPLIT_PATTERN.extract(config, defaults.splitPattern());
        separator = SEPARATOR.extract(config, defaults.separator());
        preservedValues =
            PRESERVED_VALUES.extract(config, defaults.preservedValues());
        obfuscator = OBFUSCATOR.extract(config, defaults.obfuscator());
        pattern = PATTERN.extract(config, defaults.pattern());
        section = config.sectionOrNull("replacement");
        if (section == null) {
            replacement = new LinkedHashMap<>(defaults.replacement());
        } else {
            replacement = new LinkedHashMap<>();
            for (String key: section.keys()) {
                replacement.put(key, section.getString(key));
            }
        }
        minValue = loadUnits(config, "min-value", defaults.minValue());
        maxValue = loadUnits(config, "max-value", defaults.maxValue());

        subpattern = SUBPATTERN.extract(config, defaults.subpattern());
        section = config.sectionOrNull("subreplacement");
        if (section == null) {
            subreplacement = new LinkedHashMap<>(defaults.subreplacement());
        } else {
            subreplacement = new LinkedHashMap<>();
            for (String key: section.keys()) {
                subreplacement.put(key, section.getString(key));
            }
        }
    }

    private static Map<String, Double> loadUnits(
        final IniConfig config,
        final String sectionName,
        final Map<String, Double> defaults)
        throws ConfigException
    {
        Map<String, Double> map;
        IniConfig section = config.sectionOrNull(sectionName);
        if (section == null) {
            map = new HashMap<>(defaults);
        } else {
            Double pxValue = section.get(PX, null, PARSER);
            if (pxValue == null) {
                map = new HashMap<>();
            } else {
                map = initMap(pxValue);
            }
            for (String key: section.keys()) {
                if (!key.equals(PX)) {
                    map.put(key, section.get(key, PARSER));
                }
            }
        }
        return map;
    }

    public static Map<String, Double> initMap(final Double pxValue) {
        double pxSize = pxValue.doubleValue();
        double inSize = pxSize / PX_IN_INCH;
        double cmSize = inSize * CM_IN_INCH;
        double mmSize = cmSize * MM_IN_CM;
        double ptSize = inSize * PT_IN_INCH;
        double pcSize = ptSize / PT_IN_PC;
        double emSize = pxSize / PX_IN_EM;
        double pctSize = emSize * PCT_IN_EM;
        // exotic units
        double exSize = pxSize / PX_IN_EX;
        double chSize = pxSize / PX_IN_CH;
        double remSize = pxSize / PX_IN_REM;

        Map<String, Double> map = new HashMap<>();
        map.put(PX, pxValue);
        map.put("in", inSize);
        map.put("cm", cmSize);
        map.put("mm", mmSize);
        map.put("pt", ptSize);
        map.put("pc", pcSize);
        map.put("em", emSize);
        map.put("pct", pctSize);
        // exotic units
        map.put("ex", exSize);
        map.put("ch", chSize);
        map.put("rem", remSize);
        return map;
    }

    @Override
    public UrlConfigBuilder urlConfig() {
        return urlConfig;
    }

    public T urlConfig(final UrlConfig urlConfig) {
        if (urlConfig == null) {
            this.urlConfig = null;
        } else {
            this.urlConfig = new UrlConfigBuilder(urlConfig);
        }
        return self();
    }

    @Override
    public boolean inherited() {
        return inherited;
    }

    public T inherited(final boolean inherited) {
        this.inherited = inherited;
        return self();
    }

    @Override
    public boolean trim() {
        return trim;
    }

    public T trim(final boolean trim) {
        this.trim = trim;
        return self();
    }

    @Override
    public boolean lowercase() {
        return lowercase;
    }

    public T lowercase(final boolean lowercase) {
        this.lowercase = lowercase;
        return self();
    }

    @Override
    public Pattern splitPattern() {
        return splitPattern;
    }

    public T splitPattern(final Pattern splitPattern) {
        this.splitPattern = splitPattern;
        return self();
    }

    @Override
    public String separator() {
        return separator;
    }

    public T separator(final String separator) {
        this.separator = separator;
        return self();
    }

    @Override
    public Set<String> preservedValues() {
        return preservedValues;
    }

    public T preservedValues(final Set<String> preservedValues) {
        this.preservedValues = new HashSet<>(preservedValues);
        return self();
    }

    @Override
    public Function<String, String> obfuscator() {
        return obfuscator;
    }

    public T obfuscator(final Function<String, String> obfuscator) {
        this.obfuscator = obfuscator;
        return self();
    }

    @Override
    public Pattern pattern() {
        return pattern;
    }

    public T pattern(final Pattern pattern) {
        this.pattern = pattern;
        return self();
    }

    @Override
    public Map<String, String> replacement() {
        return replacement;
    }

    public T replacement(final Map<String, String> replacement) {
        this.replacement = new LinkedHashMap<>(replacement);
        return self();
    }

    @Override
    public Map<String, Double> minValue() {
        return minValue;
    }

    public T minValue(final Map<String, Double> minValue) {
        this.minValue = new HashMap<>(minValue);
        return self();
    }

    @Override
    public Map<String, Double> maxValue() {
        return maxValue;
    }

    public T maxValue(final Map<String, Double> maxValue) {
        this.maxValue = new HashMap<>(maxValue);
        return self();
    }

    @Override
    public Pattern subpattern() {
        return subpattern;
    }

    public T subpattern(final Pattern subpattern) {
        this.subpattern = subpattern;
        return self();
    }

    @Override
    public Map<String, String> subreplacement() {
        return subreplacement;
    }

    public T subreplacement(final Map<String, String> subreplacement) {
        this.subreplacement = new LinkedHashMap<>(subreplacement);
        return self();
    }
}

