package ru.yandex.sanitizer2.config;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractSanitizingConfigBuilder
    <T extends AbstractSanitizingConfigBuilder<T>>
    implements SanitizingConfig
{
    private UrlSanitizingConfigBuilder urlSanitizingConfig;
    private boolean textUrlencoded;
    private boolean preserveDoubleBraces;
    private boolean complexCssMatching;
    private boolean foldProperties;
    private boolean detectPhishingHosts;
    private boolean wrapPlainLinks;
    private boolean compactHtml;
    private Map<String, PropertyConfigBuilder> globalAttrs;
    private Map<String, TagConfigBuilder> tags;
    private Map<String, PropertyConfigBuilder> style;
    private Map<String, ShorthandPropertyConfigBuilder> shorthands;
    private Map<String, PageHeaderConfigBuilder> pageHeaders;

    protected AbstractSanitizingConfigBuilder(final SanitizingConfig config) {
        urlSanitizingConfig(config.urlSanitizingConfig());
        textUrlencoded(config.textUrlencoded());
        preserveDoubleBraces(config.preserveDoubleBraces());
        complexCssMatching(config.complexCssMatching());
        foldProperties(config.foldProperties());
        detectPhishingHosts(config.detectPhishingHosts());
        wrapPlainLinks(config.wrapPlainLinks());
        compactHtml(config.compactHtml());
        globalAttrs(config.globalAttrs());
        tags(config.tags());
        style(config.style());
        shorthands(config.shorthands());
        pageHeaders(config.pageHeaders());
    }

    protected AbstractSanitizingConfigBuilder(
        final IniConfig config,
        final SanitizingConfig defaults)
        throws ConfigException
    {
        urlSanitizingConfig = new UrlSanitizingConfigBuilder(
            config.section("urls"),
            defaults.urlSanitizingConfig());
        IniConfig sanitizingSection = config.section("sanitizing");
        textUrlencoded = TEXT_URLENCODED.extract(
            sanitizingSection,
            defaults.textUrlencoded());
        preserveDoubleBraces = PRESERVE_DOUBLE_BRACES.extract(
            sanitizingSection,
            defaults.preserveDoubleBraces());
        complexCssMatching = COMPLEX_CSS_MATCHING.extract(
            sanitizingSection,
            defaults.complexCssMatching());
        foldProperties = FOLD_PROPERTIES.extract(
            sanitizingSection,
            defaults.foldProperties());
        detectPhishingHosts = DETECT_PHISHING_HOSTS.extract(
            sanitizingSection,
            defaults.detectPhishingHosts());
        wrapPlainLinks = WRAP_PLAIN_LINKS.extract(
            sanitizingSection,
            defaults.wrapPlainLinks());
        compactHtml = COMPACT_HTML.extract(
            sanitizingSection,
            defaults.compactHtml());

        // Global attrs defaults ignored
        Map<String, IniConfig> sections =
            config.section("global-attr").sections();
        globalAttrs = new HashMap<>(sections.size() << 1);
        for (Map.Entry<String, IniConfig> entry: sections.entrySet()) {
            globalAttrs.put(
                entry.getKey(),
                new PropertyConfigBuilder(entry.getValue()));
        }
        // Tags defaults ignored
        sections = config.section("tag").sections();
        tags = new HashMap<>(sections.size() << 1);
        for (Map.Entry<String, IniConfig> entry: sections.entrySet()) {
            tags.put(
                entry.getKey(),
                new TagConfigBuilder(entry.getValue()));
        }
        // Styles defaults ignored
        sections = config.section("style").sections();
        style = new HashMap<>(sections.size() << 1);
        for (Map.Entry<String, IniConfig> entry: sections.entrySet()) {
            style.put(
                entry.getKey(),
                new PropertyConfigBuilder(entry.getValue()));
        }
        // Shorthand properties defaults ignored
        sections = config.section("shorthand-property").sections();
        shorthands = new HashMap<>(sections.size() << 1);
        for (Map.Entry<String, IniConfig> entry: sections.entrySet()) {
            shorthands.put(
                entry.getKey(),
                new ShorthandPropertyConfigBuilder(entry.getValue()));
        }
        // Page headers defaults ignored
        sections = config.section("page-header").sections();
        pageHeaders = new LinkedHashMap<>(sections.size() << 1);
        for (Map.Entry<String, IniConfig> entry: sections.entrySet()) {
            pageHeaders.put(
                entry.getKey(),
                new PageHeaderConfigBuilder(entry.getValue()));
        }
    }

    protected abstract T self();

    @Override
    public UrlSanitizingConfigBuilder urlSanitizingConfig() {
        return urlSanitizingConfig;
    }

    public T urlSanitizingConfig(
        final UrlSanitizingConfig urlSanitizingConfig)
    {
        this.urlSanitizingConfig =
            new UrlSanitizingConfigBuilder(urlSanitizingConfig);
        return self();
    }

    @Override
    public boolean textUrlencoded() {
        return textUrlencoded;
    }

    public T textUrlencoded(final boolean textUrlencoded) {
        this.textUrlencoded = textUrlencoded;
        return self();
    }

    @Override
    public boolean preserveDoubleBraces() {
        return preserveDoubleBraces;
    }

    public T preserveDoubleBraces(final boolean preserveDoubleBraces) {
        this.preserveDoubleBraces = preserveDoubleBraces;
        return self();
    }

    @Override
    public boolean complexCssMatching() {
        return complexCssMatching;
    }

    public T complexCssMatching(final boolean complexCssMatching) {
        this.complexCssMatching = complexCssMatching;
        return self();
    }

    @Override
    public boolean foldProperties() {
        return foldProperties;
    }

    public T foldProperties(final boolean foldProperties) {
        this.foldProperties = foldProperties;
        return self();
    }

    @Override
    public boolean detectPhishingHosts() {
        return detectPhishingHosts;
    }

    public T detectPhishingHosts(final boolean detectPhishingHosts) {
        this.detectPhishingHosts = detectPhishingHosts;
        return self();
    }

    @Override
    public boolean wrapPlainLinks() {
        return wrapPlainLinks;
    }

    public T wrapPlainLinks(final boolean wrapPlainLinks) {
        this.wrapPlainLinks = wrapPlainLinks;
        return self();
    }

    @Override
    public boolean compactHtml() {
        return compactHtml;
    }

    public T compactHtml(final boolean compactHtml) {
        this.compactHtml = compactHtml;
        return self();
    }

    @Override
    public Map<String, PropertyConfigBuilder> globalAttrs() {
        return globalAttrs;
    }

    public T globalAttrs(
        final Map<String, ? extends PropertyConfig> globalAttrs)
    {
        this.globalAttrs = new HashMap<>(globalAttrs.size() << 1);
        for (Map.Entry<String, ? extends PropertyConfig> entry
            : globalAttrs.entrySet())
        {
            this.globalAttrs.put(
                entry.getKey(),
                new PropertyConfigBuilder(entry.getValue()));
        }
        return self();
    }

    @Override
    public Map<String, TagConfigBuilder> tags() {
        return tags;
    }

    public T tags(final Map<String, ? extends TagConfig> tags) {
        this.tags = new HashMap<>(tags.size() << 1);
        for (Map.Entry<String, ? extends TagConfig> entry
            : tags.entrySet())
        {
            this.tags.put(
                entry.getKey(),
                new TagConfigBuilder(entry.getValue()));
        }
        return self();
    }

    @Override
    public Map<String, PropertyConfigBuilder> style() {
        return style;
    }

    public T style(final Map<String, ? extends PropertyConfig> style) {
        this.style = new HashMap<>(style.size() << 1);
        for (Map.Entry<String, ? extends PropertyConfig> entry
            : style.entrySet())
        {
            this.style.put(
                entry.getKey(),
                new PropertyConfigBuilder(entry.getValue()));
        }
        return self();
    }

    @Override
    public Map<String, ShorthandPropertyConfigBuilder> shorthands() {
        return shorthands;
    }

    public T shorthands(
        final Map<String, ? extends ShorthandPropertyConfig> shorthands)
    {
        this.shorthands = new HashMap<>(shorthands.size() << 1);
        for (Map.Entry<String, ? extends ShorthandPropertyConfig> entry
            : shorthands.entrySet())
        {
            this.shorthands.put(
                entry.getKey(),
                new ShorthandPropertyConfigBuilder(entry.getValue()));
        }
        return self();
    }

    @Override
    public Map<String, PageHeaderConfigBuilder> pageHeaders() {
        return pageHeaders;
    }

    public T pageHeaders(
        final Map<String, ? extends PageHeaderConfig> pageHeaders)
    {
        this.pageHeaders = new LinkedHashMap<>(pageHeaders.size() << 1);
        for (Map.Entry<String, ? extends PageHeaderConfig> entry
            : pageHeaders.entrySet())
        {
            this.pageHeaders.put(
                entry.getKey(),
                new PageHeaderConfigBuilder(entry.getValue()));
        }
        return self();
    }
}

