package ru.yandex.sanitizer2.config;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.collection.IntPair;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.ImmutableConfig;
import ru.yandex.sanitizer2.AttributeSanitizer;
import ru.yandex.sanitizer2.SanitizingContext;
import ru.yandex.sanitizer2.UrlRewriter;

public class ImmutablePropertyConfig
    implements AttributeSanitizer, ImmutableConfig, PropertyConfig
{
    private final ImmutableUrlConfig urlConfig;
    private final boolean rewriteUrlOnSanitize;
    private final UrlRewriter urlRewriter;
    private final boolean inherited;
    private final boolean trim;
    private final boolean lowercase;
    private final boolean dropPathAndQuery;
    private final Pattern splitPattern;
    private final String separator;
    private final Set<String> preservedValues;
    private final Function<String, String> obfuscator;
    private final Pattern pattern;
    private final Map<String, String> replacement;
    private final Replacement[] replacementArray;
    private final Map<String, Double> minValue;
    private final ValueConstraint[] minValueArray;
    private final Map<String, Double> maxValue;
    private final ValueConstraint[] maxValueArray;
    private final Pattern subpattern;
    private final Map<String, String> subreplacement;
    private final Replacement[] subreplacementArray;

    private static final Pattern FRAGMENT =
        Pattern.compile("#[^#]*$");
    private static final Pattern PROTOCOL_AND_AUTHORITY =
        Pattern.compile("^((\\p{Lower}*:)?//)?[^/?]*[/?]?");

    public ImmutablePropertyConfig(
        final PropertyConfig config,
        final ImmutableUrlSanitizingConfig urlSanitizingConfig,
        final boolean rewriteUrlOnSanitize)
        throws ConfigException
    {
        UrlConfig urlConfig = config.urlConfig();
        if (urlConfig == null) {
            this.urlConfig = null;
            this.rewriteUrlOnSanitize = false;
            urlRewriter = null;
        } else {
            this.urlConfig = new ImmutableUrlConfig(urlConfig);
            this.rewriteUrlOnSanitize = rewriteUrlOnSanitize;
            urlRewriter =
                UrlRewriter.create(urlSanitizingConfig, this.urlConfig);
        }
        inherited = INHERITED.validate(config.inherited());
        trim = TRIM.validate(config.trim());
        lowercase = LOWERCASE.validate(config.lowercase());
        splitPattern = SPLIT_PATTERN.validate(config.splitPattern());
        separator = SEPARATOR.validate(config.separator());
        preservedValues =
            CollectionCompactor.compact(
                new HashSet<>(
                    PRESERVED_VALUES.validate(config.preservedValues())));
        obfuscator = OBFUSCATOR.validate(config.obfuscator());
        pattern = PATTERN.validate(config.pattern());
        Map<String, String> replacement = config.replacement();
        int replacementSize = replacement.size();
        Map<String, String> replacementCopy =
            new LinkedHashMap<>(replacementSize << 1);
        replacementArray = new Replacement[replacementSize];
        int i = 0;
        for (Map.Entry<String, String> entry: replacement.entrySet()) {
            String group = entry.getKey().intern();
            String value = entry.getValue().intern();
            replacementCopy.put(group, value);
            replacementArray[i++] = new Replacement(group, value);
        }
        this.replacement = CollectionCompactor.compact(replacementCopy);
        minValue = copy(config.minValue());
        minValueArray = toValueConstraints(minValue);
        maxValue = copy(config.maxValue());
        maxValueArray = toValueConstraints(maxValue);

        subpattern = SUBPATTERN.validate(config.subpattern());
        Map<String, String> subreplacement = config.subreplacement();
        int subreplacementSize = subreplacement.size();
        Map<String, String> subreplacementCopy =
            new LinkedHashMap<>(subreplacementSize << 1);
        subreplacementArray = new Replacement[subreplacementSize];
        i = 0;
        for (Map.Entry<String, String> entry: subreplacement.entrySet()) {
            String group = entry.getKey().intern();
            String value = entry.getValue().intern();
            subreplacementCopy.put(group, value);
            subreplacementArray[i++] = new Replacement(group, value);
        }
        this.subreplacement = CollectionCompactor.compact(subreplacementCopy);

        if (splitPattern == null && separator != null) {
            throw new ConfigException(
                "Separator is set but no split-pattern defined");
        }

        if (pattern == null) {
            if (!replacement.isEmpty()) {
                throw new ConfigException(
                    "No pattern is set, but replacement presents");
            } else if (minValue.size() + maxValue.size() > 0) {
                throw new ConfigException(
                    "No pattern is set, but value constraints present");
            }
        } else {
            int size =
                replacementSize + minValue.size() + maxValue.size();
            if (size > 0) {
                Set<String> namedGroups = new HashSet<>(size << 1);
                namedGroups.addAll(replacement.keySet());
                namedGroups.addAll(minValue.keySet());
                namedGroups.addAll(maxValue.keySet());
                checkCapturingGroups(pattern, namedGroups);
            }
        }

        if (subpattern == null) {
            if (!subreplacement.isEmpty()) {
                throw new ConfigException(
                    "No subpattern is set, but subreplacement presents");
            }
        } else if (subreplacement.isEmpty()) {
            throw new ConfigException(
                "Subpattern is set, but subreplacement doesn't present");
        } else {
            checkCapturingGroups(subpattern, subreplacement.keySet());
        }

        dropPathAndQuery = urlSanitizingConfig.dropPathAndQuery();
    }

    private static Map<String, Double> copy(final Map<String, Double> map)
        throws ConfigException
    {
        Map<String, Double> result = new HashMap<>(map.size() << 1);
        for (Map.Entry<String, Double> entry: map.entrySet()) {
            String unit = entry.getKey();
            Double value = entry.getValue();
            if (Double.isFinite(value.doubleValue())) {
                result.put(unit.intern(), value);
            } else {
                throw new ConfigException(
                    "Invalid value for unit '" + unit
                    + "': " + value);
            }
        }
        return CollectionCompactor.compact(result);
    }

    private static ValueConstraint[] toValueConstraints(
        final Map<String, Double> map)
    {
        ValueConstraint[] constraints = new ValueConstraint[map.size()];
        int i = 0;
        for (Map.Entry<String, Double> entry: map.entrySet()) {
            constraints[i++] =
                new ValueConstraint(entry.getKey(), entry.getValue());
        }
        return constraints;
    }

    private static void checkCapturingGroups(
        final Pattern pattern,
        final Set<String> groups)
        throws ConfigException
    {
        Matcher matcher =
            Pattern.compile("(?:" + pattern + ")|.").matcher(".");
        if (matcher.matches()) {
            for (String group: groups) {
                try {
                    matcher.group(group);
                } catch (RuntimeException e) {
                    throw new ConfigException(
                        "There is no capturing group <" + group
                        + "> in pattern <" + pattern + '>',
                        e);
                }
            }
        }
    }

    @Override
    public ImmutableUrlConfig urlConfig() {
        return urlConfig;
    }

    public UrlRewriter urlRewriter() {
        return urlRewriter;
    }

    @Override
    public boolean inherited() {
        return inherited;
    }

    @Override
    public boolean trim() {
        return trim;
    }

    @Override
    public boolean lowercase() {
        return lowercase;
    }

    @Override
    public Pattern splitPattern() {
        return splitPattern;
    }

    @Override
    public String separator() {
        return separator;
    }

    @Override
    public Set<String> preservedValues() {
        return preservedValues;
    }

    @Override
    public Function<String, String> obfuscator() {
        return obfuscator;
    }

    @Override
    public Pattern pattern() {
        return pattern;
    }

    @Override
    public Map<String, String> replacement() {
        return replacement;
    }

    @Override
    public Map<String, Double> minValue() {
        return minValue;
    }

    @Override
    public Map<String, Double> maxValue() {
        return maxValue;
    }

    @Override
    public Pattern subpattern() {
        return subpattern;
    }

    @Override
    public Map<String, String> subreplacement() {
        return subreplacement;
    }

    @Override
    @Nullable
    public String sanitize(
        @Nonnull final String attrValue,
        @Nonnull final SanitizingContext context)
    {
        Map<String, IntPair<String>> cache = context.propertyCache(this);
        int contextGeneration = context.contextGeneration();
        String value;
        if (splitPattern == null) {
            value = sanitizePart(attrValue, cache, contextGeneration, context);
        } else {
            Matcher matcher = splitPattern.matcher(attrValue);
            if (matcher.find()) {
                int len = attrValue.length();
                StringBuilder sb = new StringBuilder(len);
                int pos = 0;
                int end = 0;
                do {
                    int start = matcher.start();
                    if (start == 0) {
                        end = matcher.end();
                        pos = end;
                    } else {
                        String sanitized = sanitizePart(
                            attrValue.substring(end, start),
                            cache,
                            contextGeneration,
                            context);
                        if (sanitized == null) {
                            return null;
                        } else {
                            if (end > pos) {
                                if (separator == null) {
                                    sb.append(attrValue, pos, end);
                                } else {
                                    sb.append(separator);
                                }
                            }
                            sb.append(sanitized);
                            pos = start;
                            end = matcher.end();
                        }
                    }
                } while (matcher.find());
                if (end == len) {
                    // Skip trailing separator
                    value = new String(sb);
                } else {
                    String sanitized = sanitizePart(
                        attrValue.substring(end),
                        cache,
                        contextGeneration,
                        context);
                    if (sanitized == null) {
                        value = null;
                    } else {
                        if (end > pos) {
                            if (separator == null) {
                                sb.append(attrValue, pos, end);
                            } else {
                                sb.append(separator);
                            }
                        }
                        sb.append(sanitized);
                        value = new String(sb);
                    }
                }
            } else {
                value = sanitizePart(
                    attrValue,
                    cache,
                    contextGeneration,
                    context);
            }
        }
        return value;
    }

    private String sanitizePart(
        final String attrValue,
        final Map<String, IntPair<String>> cache,
        final int contextGeneration,
        final SanitizingContext context)
    {
        IntPair<String> cached = cache.get(attrValue);
        if (cached == null || cached.first() != contextGeneration) {
            String sanitized = sanitizePart(attrValue, context);
            cache.put(attrValue, new IntPair<>(contextGeneration, sanitized));
            return sanitized;
        } else {
            return cached.second();
        }
    }

    private String sanitizePart(
        final String attrValue,
        final SanitizingContext context)
    {
        String value = attrValue;
        if (trim) {
            value = value.trim();
        }
        if (lowercase) {
            value = value.toLowerCase(Locale.ROOT);
        }
        if (!preservedValues.contains(value)) {
            if (!value.isEmpty()) {
                value = obfuscator.apply(value);
            }
            if (pattern != null) {
                Matcher matcher = pattern.matcher(value);
                if (matcher.matches()) {
                    for (ValueConstraint minValue: minValueArray) {
                        String group = matcher.group(minValue.group);
                        if (group != null) {
                            try {
                                group = group.replace(',', '.');
                                double doubleValue = Double.parseDouble(group);
                                if (doubleValue < minValue.constraint) {
                                    value = null;
                                }
                            } catch (RuntimeException e) {
                                value = null;
                            }
                            break;
                        }
                    }
                    if (value != null) {
                        for (ValueConstraint maxValue: maxValueArray) {
                            String group = matcher.group(maxValue.group);
                            if (group != null) {
                                try {
                                    group = group.replace(',', '.');
                                    double doubleValue =
                                        Double.parseDouble(group);
                                    if (doubleValue > maxValue.constraint) {
                                        value = null;
                                    }
                                } catch (RuntimeException e) {
                                    value = null;
                                }
                                break;
                            }
                        }
                        if (value != null) {
                            for (Replacement replacement: replacementArray) {
                                String group = matcher.group(replacement.group);
                                if (group != null) {
                                    StringBuilder sb = new StringBuilder();
                                    matcher.appendReplacement(
                                        sb,
                                        replacement.value);
                                    matcher.appendTail(sb);
                                    value = sb.toString();
                                    break;
                                }
                            }
                            if (subpattern != null) {
                                matcher = subpattern.matcher(value);
                                if (matcher.find()) {
                                    StringBuilder sb = new StringBuilder();
                                    do {
                                        for (Replacement replacement
                                            : subreplacementArray)
                                        {
                                            String group = matcher.group(
                                                replacement.group);
                                            if (group != null) {
                                                matcher.appendReplacement(
                                                    sb,
                                                    replacement.value);
                                                break;
                                            }
                                        }
                                    } while (matcher.find());
                                    matcher.appendTail(sb);
                                    value = sb.toString();
                                }
                            }
                        }
                    }
                } else {
                    value = null;
                }
            }
            if (value != null && rewriteUrlOnSanitize && urlRewriter != null) {
                value = urlRewriter.sanitize(value, context);
                if (dropPathAndQuery) {
                    value = doDropPathAndQuery(value);
                }
            }
        }
        return value;
    }

    /**
    * Sould be UrlRewriter, but FastUrlParser is not complete:
    * mailto:a@b return error. Logic for differentiating
    * ya.ru:80 from tel:3242 is not trivial,
    *                       but doesn't required in this crude approach
     */
    private static String doDropPathAndQuery(@Nullable String uri) {
        if (uri == null || uri.isEmpty()) {
            return uri;
        }
        Matcher fragmentMatcher = FRAGMENT.matcher(uri);
        Matcher prefixMatcher = PROTOCOL_AND_AUTHORITY.matcher(uri);
        String fragment = "";
        if (fragmentMatcher.find()) {
            fragment = fragmentMatcher.group();
            prefixMatcher.region(0, uri.length() - fragment.length());
        }
        if (!prefixMatcher.find()) {
            return fragment;
        }
        return prefixMatcher.group() + fragment;
    }

    private static class Replacement {
        private final String group;
        private final String value;

        Replacement(final String group, final String value) {
            this.group = group;
            this.value = value;
        }
    }

    private static class ValueConstraint {
        private final String group;
        private final double constraint;

        ValueConstraint(final String group, final double constraint) {
            this.group = group;
            this.constraint = constraint;
        }
    }
}

