package ru.yandex.sanitizer2.config;

import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.owasp.html.AttributePolicy;
import org.owasp.html.HtmlPolicyBuilder;
import org.owasp.html.PolicyFactory;

import ru.yandex.parser.config.ConfigException;

public class ImmutableSanitizingConfig
    extends Interner
    implements SanitizingConfig
{
    private final ImmutableUrlSanitizingConfig urlSanitizingConfig;
    private final boolean textUrlencoded;
    private final boolean preserveDoubleBraces;
    private final boolean complexCssMatching;
    private final boolean foldProperties;
    private final boolean detectPhishingHosts;
    private final boolean wrapPlainLinks;
    private final boolean compactHtml;
    private final IdentityHashMap<String, ImmutablePropertyConfig> globalAttrs;
    private final IdentityHashMap<String, ImmutableTagConfig> tags;
    private final IdentityHashMap<String, ImmutablePropertyConfig> style;
    private final IdentityHashMap<String, ImmutableShorthandPropertyConfig>
        shorthands;
    private final Map<String, ImmutablePageHeaderConfig> pageHeaders;
    private final IdentityHashMap<String, Fold> folds;
    private final String styleTag;
    private final String baseTag;
    private final String aTag;
    private final String brTag;
    private final String blockquoteTag;
    private final String pTag;
    private final String spanTag;
    private final String classAttr;
    private final String idAttr;
    private final String styleAttr;
    private final String hrefAttr;
    private final ImmutablePropertyConfig classConfig;
    private final ImmutablePropertyConfig idConfig;

    public ImmutableSanitizingConfig(final SanitizingConfig config)
        throws ConfigException
    {
        urlSanitizingConfig =
            new ImmutableUrlSanitizingConfig(config.urlSanitizingConfig());
        textUrlencoded = TEXT_URLENCODED.validate(config.textUrlencoded());
        preserveDoubleBraces =
            PRESERVE_DOUBLE_BRACES.validate(config.preserveDoubleBraces());
        complexCssMatching =
            COMPLEX_CSS_MATCHING.validate(config.complexCssMatching());
        foldProperties = FOLD_PROPERTIES.validate(config.foldProperties());
        detectPhishingHosts =
            DETECT_PHISHING_HOSTS.validate(config.detectPhishingHosts());
        wrapPlainLinks = WRAP_PLAIN_LINKS.validate(config.wrapPlainLinks());
        compactHtml = COMPACT_HTML.validate(config.compactHtml());

        Map<String, ? extends PropertyConfig> globalAttrs =
            config.globalAttrs();
        this.globalAttrs = new IdentityHashMap<>(globalAttrs.size() << 1);
        for (Map.Entry<String, ? extends PropertyConfig> entry
            : globalAttrs.entrySet())
        {
            String attr = entry.getKey().intern();
            addAttr(attr);
            this.globalAttrs.put(
                attr,
                new ImmutablePropertyConfig(
                    entry.getValue(),
                    urlSanitizingConfig,
                    true));
        }
        Map<String, ? extends TagConfig> tags = config.tags();
        this.tags = new IdentityHashMap<>(tags.size() << 1);
        for (Map.Entry<String, ? extends TagConfig> entry
            : tags.entrySet())
        {
            String tag = entry.getKey().intern();
            addTag(tag);
            ImmutableTagConfig tagConfig =
                new ImmutableTagConfig(entry.getValue(), urlSanitizingConfig);
            tagConfig.populateInterner(this);
            this.tags.put(tag, tagConfig);
        }
        Map<String, ? extends PropertyConfig> style = config.style();
        this.style = new IdentityHashMap<>(style.size() << 1);
        for (Map.Entry<String, ? extends PropertyConfig> entry
            : style.entrySet())
        {
            String property = entry.getKey().intern();
            addProperty(property);
            this.style.put(
                property,
                new ImmutablePropertyConfig(
                    entry.getValue(),
                    urlSanitizingConfig,
                    false));
        }

        Map<String, ? extends ShorthandPropertyConfig> shorthands =
            config.shorthands();
        this.shorthands = new IdentityHashMap<>(shorthands.size() << 1);
        folds = new IdentityHashMap<>((shorthands.size() + style.size()) << 1);
        for (Map.Entry<String, ? extends ShorthandPropertyConfig> entry
            : shorthands.entrySet())
        {
            String shorthandName = entry.getKey().intern();
            addProperty(shorthandName);
            ImmutableShorthandPropertyConfig shorthand =
                new ImmutableShorthandPropertyConfig(entry.getValue());
            for (ImmutablePropertyAliasConfig properties
                : shorthand.properties().values())
            {
                for (String propertyName: properties.names()) {
                    if (!style.containsKey(propertyName)
                        && !shorthands.containsKey(propertyName))
                    {
                        throw new ConfigException(
                            "Shorthand " + shorthandName
                            + " referes to unknown property " + propertyName);
                    }
                }
            }
            ImmutablePropertyFoldsConfig foldsConfig = shorthand.foldsConfig();
            Set<String> mandatoryProperties = new HashSet<>();
            for (Map.Entry<String, ImmutablePropertyFoldConfig> foldEntry
                : foldsConfig.properties().entrySet())
            {
                String name = foldEntry.getKey();
                ImmutablePropertyFoldConfig foldConfig = foldEntry.getValue();
                if (foldConfig.mandatory()
                    || foldConfig.defaultValue() == null)
                {
                    mandatoryProperties.add(name);
                } else {
                    PropertyConfig propertyConfig = style.get(name);
                    if (propertyConfig == null) {
                        if (shorthands.get(name).inherited()) {
                            mandatoryProperties.add(name);
                        }
                    } else if (propertyConfig.inherited()) {
                        mandatoryProperties.add(name);
                    }
                }
            }
            for (Map.Entry<String, ImmutablePropertyFoldConfig> foldEntry
                : foldsConfig.properties().entrySet())
            {
                String foldName = foldEntry.getKey();
                if (!style.containsKey(foldName)
                    && !shorthands.containsKey(foldName))
                {
                    throw new ConfigException(
                        "Unknown fold property: " + foldName);
                }
                ImmutablePropertyFoldConfig foldConfig = foldEntry.getValue();
                String requiredProperty = foldConfig.requiredProperty();
                if (requiredProperty != null
                    && !style.containsKey(requiredProperty)
                    && !shorthands.containsKey(requiredProperty))
                {
                    throw new ConfigException(
                        "Unknown required fold property: " + requiredProperty);
                }
                Fold oldFold = folds.put(
                    foldName,
                    new Fold(
                        shorthandName,
                        mandatoryProperties.size(),
                        mandatoryProperties.contains(foldName),
                        foldConfig));
                if (oldFold != null) {
                    throw new ConfigException(
                        "Fold " + foldName
                        + " used by both " + shorthandName
                        + " and " + oldFold.shorthandName + " shorthands");
                }
            }
            // Interner will ensure that all property names properly interned
            shorthand.populateInterner(this);
            this.shorthands.put(shorthandName, shorthand);
        }
        Map<String, ? extends PageHeaderConfig> pageHeaders =
            config.pageHeaders();
        this.pageHeaders = new LinkedHashMap<>(pageHeaders.size() << 1);
        for (Map.Entry<String, ? extends PageHeaderConfig> entry
            : pageHeaders.entrySet())
        {
            this.pageHeaders.put(
                entry.getKey(),
                new ImmutablePageHeaderConfig(entry.getValue()));
        }

        styleTag = internTag("style");
        baseTag = internTag("base");
        aTag = internTag("a");
        brTag = internTag("br");
        blockquoteTag = internTag("blockquote");
        pTag = internTag("p");
        spanTag = internTag("span");
        classAttr = internAttr("class");
        idAttr = internAttr("id");
        styleAttr = internAttr("style");
        hrefAttr = internAttr("href");
        classConfig = this.globalAttrs.get(classAttr);
        idConfig = this.globalAttrs.get(idAttr);
    }

    @Override
    public ImmutableUrlSanitizingConfig urlSanitizingConfig() {
        return urlSanitizingConfig;
    }

    @Override
    public boolean textUrlencoded() {
        return textUrlencoded;
    }

    @Override
    public boolean preserveDoubleBraces() {
        return preserveDoubleBraces;
    }

    @Override
    public boolean complexCssMatching() {
        return complexCssMatching;
    }

    @Override
    public boolean foldProperties() {
        return foldProperties;
    }

    @Override
    public boolean detectPhishingHosts() {
        return detectPhishingHosts;
    }

    @Override
    public boolean wrapPlainLinks() {
        return wrapPlainLinks;
    }

    @Override
    public boolean compactHtml() {
        return compactHtml;
    }

    @Override
    public Map<String, ImmutablePropertyConfig> globalAttrs() {
        return globalAttrs;
    }

    @Override
    public Map<String, ImmutableTagConfig> tags() {
        return tags;
    }

    @Override
    public Map<String, ImmutablePropertyConfig> style() {
        return style;
    }

    @Override
    public Map<String, ImmutableShorthandPropertyConfig> shorthands() {
        return shorthands;
    }

    @Override
    public Map<String, ImmutablePageHeaderConfig> pageHeaders() {
        return pageHeaders;
    }

    public Map<String, Fold> folds() {
        return folds;
    }

    public String styleTag() {
        return styleTag;
    }

    public String baseTag() {
        return baseTag;
    }

    public String aTag() {
        return aTag;
    }

    public String brTag() {
        return brTag;
    }

    public String blockquoteTag() {
        return blockquoteTag;
    }

    public String pTag() {
        return pTag;
    }

    public String spanTag() {
        return spanTag;
    }

    public String classAttr() {
        return classAttr;
    }

    public String idAttr() {
        return idAttr;
    }

    public String styleAttr() {
        return styleAttr;
    }

    public String hrefAttr() {
        return hrefAttr;
    }

    public ImmutablePropertyConfig classConfig() {
        return classConfig;
    }

    public ImmutablePropertyConfig idConfig() {
        return idConfig;
    }

    public PolicyFactory createPolicyFactory() {
        HtmlPolicyBuilder htmlPolicyBuilder =
            new HtmlPolicyBuilder()
                .allowUrlsInStyles(AttributePolicy.IDENTITY_ATTRIBUTE_POLICY)
                .allowTextIn("style", "table")
                .disableUrlGuards()
                .disableRelsOnLinksPolicy();
        for (String tagName: HtmlPolicyBuilder.DEFAULT_SKIP_IF_EMPTY) {
            htmlPolicyBuilder.allowWithoutAttributes(tagName);
        }
        Set<String> bypassSchemes = urlSanitizingConfig.bypassSchemes();
        htmlPolicyBuilder.allowUrlProtocols(
            bypassSchemes.toArray(new String[bypassSchemes.size()]));
        Set<String> allowedSchemes = urlSanitizingConfig.allowedSchemes();
        htmlPolicyBuilder.allowUrlProtocols(
            allowedSchemes.toArray(new String[allowedSchemes.size()]));
        for (Map.Entry<String, ImmutablePropertyConfig> entry
            : globalAttrs.entrySet())
        {
            htmlPolicyBuilder
                .allowAttributes(entry.getKey())
                .globally();
        }
        for (Map.Entry<String, ImmutableTagConfig> entry
            : tags.entrySet())
        {
            String tagName = entry.getKey();
            htmlPolicyBuilder.allowElements(tagName);
            ImmutableTagConfig tagConfig = entry.getValue();
            IdentityHashMap<String, ImmutablePropertyConfig> attrs =
                tagConfig.attrs();
            if (!attrs.isEmpty()) {
                for (Map.Entry<String, ImmutablePropertyConfig> attrEntry
                    : attrs.entrySet())
                {
                    htmlPolicyBuilder
                        .allowAttributes(attrEntry.getKey())
                        .onElements(tagName);
                }
            }
        }
        return htmlPolicyBuilder.toFactory();
    }

    public static class Fold {
        private final String shorthandName;
        private final int mandatoryPropertiesCount;
        private final boolean mandatory;
        private final boolean allowSpaces;
        private final boolean allowCommas;
        private final String requiredProperty;

        public Fold(
            final String shorthandName,
            final int mandatoryPropertiesCount,
            final boolean mandatory,
            final ImmutablePropertyFoldConfig config)
        {
            this.shorthandName = shorthandName;
            this.mandatoryPropertiesCount = mandatoryPropertiesCount;
            this.mandatory = mandatory;
            allowSpaces = config.allowSpaces();
            allowCommas = config.allowCommas();
            requiredProperty = config.requiredProperty();
        }

        public String shorthandName() {
            return shorthandName;
        }

        public int mandatoryPropertiesCount() {
            return mandatoryPropertiesCount;
        }

        public boolean mandatory() {
            return mandatory;
        }

        public boolean allowSpaces() {
            return allowSpaces;
        }

        public boolean allowCommas() {
            return allowCommas;
        }

        public String requiredProperty() {
            return requiredProperty;
        }
    }
}

