package ru.yandex.sanitizer2.config;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.string.PatternGroupsExtractor;

public class ImmutableShorthandPropertyConfig
    implements ShorthandPropertyConfig
{
    private final boolean inherited;
    private final Pattern pattern;
    private final Map<String, ImmutablePropertyAliasConfig> properties;
    private final List<Shorthand> propertiesList;
    private final ImmutablePropertyFoldsConfig foldsConfig;

    public ImmutableShorthandPropertyConfig(
        final ShorthandPropertyConfig config)
        throws ConfigException
    {
        inherited = INHERITED.validate(config.inherited());
        pattern = PATTERN.validate(config.pattern());

        Map<String, ? extends PropertyAliasConfig> properties =
            config.properties();
        int size = properties.size();
        this.properties = new LinkedHashMap<>(size << 1);
        propertiesList = new ArrayList<>(size);
        for (Map.Entry<String, ? extends PropertyAliasConfig> entry
            : properties.entrySet())
        {
            String property = entry.getKey().intern();
            ImmutablePropertyAliasConfig aliasConfig =
                new ImmutablePropertyAliasConfig(entry.getValue());
            this.properties.put(property, aliasConfig);
            propertiesList.add(new Shorthand(property, aliasConfig));
        }
        Set<String> groups = PatternGroupsExtractor.INSTANCE.apply(pattern);
        Set<String> propertiesSet = properties.keySet();
        if (!groups.equals(propertiesSet)) {
            throw new ConfigException(
                "Pattern groups differs from properties set: "
                + groups + " != " + propertiesSet);
        }

        foldsConfig = new ImmutablePropertyFoldsConfig(config.foldsConfig());
    }

    @Override
    public boolean inherited() {
        return inherited;
    }

    @Override
    public Pattern pattern() {
        return pattern;
    }

    @Override
    public Map<String, ImmutablePropertyAliasConfig> properties() {
        return properties;
    }

    @Override
    public ImmutablePropertyFoldsConfig foldsConfig() {
        return foldsConfig;
    }

    public List<Shorthand> propertiesList() {
        return propertiesList;
    }

    public void populateInterner(final Interner interner) {
        for (Shorthand shorthand: propertiesList) {
            shorthand.aliasConfig.populateInterner(interner);
        }
        foldsConfig.populateInterner(interner);
    }

    public static class Shorthand {
        private final String property;
        private final ImmutablePropertyAliasConfig aliasConfig;

        public Shorthand(
            final String property,
            final ImmutablePropertyAliasConfig aliasConfig)
        {
            this.property = property;
            this.aliasConfig = aliasConfig;
        }

        public String property() {
            return property;
        }

        public ImmutablePropertyAliasConfig aliasConfig() {
            return aliasConfig;
        }
    }
}

