package ru.yandex.sanitizer2.config;

import java.util.IdentityHashMap;
import java.util.Map;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.util.string.StringUtils;

public class ImmutableTagConfig implements TagConfig {
    private final String renameTo;
    private final boolean requireContent;
    private final boolean requireAttrs;
    private final boolean combine;
    private final boolean flag;
    private final boolean ignoreSelfClose;
    private final IdentityHashMap<String, ImmutablePropertyConfig> attrs;

    public ImmutableTagConfig(
        final TagConfig config,
        final ImmutableUrlSanitizingConfig urlSanitizingConfig)
        throws ConfigException
    {
        renameTo = StringUtils.intern(RENAME_TO.validate(config.renameTo()));
        requireContent = REQUIRE_CONTENT.validate(config.requireContent());
        requireAttrs = REQUIRE_ATTRS.validate(config.requireAttrs());
        combine = COMBINE.validate(config.combine());
        flag = FLAG.validate(config.flag());
        ignoreSelfClose = IGNORE_SELF_CLOSE.validate(config.ignoreSelfClose());
        Map<String, ? extends PropertyConfig> attrs = config.attrs();
        if (attrs.isEmpty()) {
            this.attrs = new IdentityHashMap<>();
        } else {
            if (flag) {
                throw new ConfigException("Flag tags should not have attrs");
            } else {
                this.attrs = new IdentityHashMap<>(attrs.size() << 1);
                for (Map.Entry<String, ? extends PropertyConfig> entry
                    : attrs.entrySet())
                {
                    String name = entry.getKey().intern();
                    this.attrs.put(
                        name,
                        TagConfig.attributeConfigParser(
                            name,
                            urlSanitizingConfig)
                            .validate(entry.getValue()));
                }
            }
        }
    }

    @Override
    public String renameTo() {
        return renameTo;
    }

    @Override
    public boolean requireContent() {
        return requireContent;
    }

    @Override
    public boolean requireAttrs() {
        return requireAttrs;
    }

    @Override
    public boolean combine() {
        return combine;
    }

    @Override
    public boolean flag() {
        return flag;
    }

    @Override
    public boolean ignoreSelfClose() {
        return ignoreSelfClose;
    }

    @Override
    public IdentityHashMap<String, ImmutablePropertyConfig> attrs() {
        return attrs;
    }

    public void populateInterner(final Interner interner) {
        if (renameTo != null) {
            interner.addTag(renameTo);
        }
        for (String attr: attrs.keySet()) {
            interner.addAttr(attr);
        }
    }
}

