package ru.yandex.sanitizer2.config;

import java.util.HashSet;
import java.util.Set;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.url.processor.ImmutableUrlProcessorConfig;
import ru.yandex.url.processor.UrlProcessorConfigBuilder;

public class ImmutableUrlSanitizingConfig implements UrlSanitizingConfig {
    private final ImmutableResizerConfig resizerConfig;
    private final boolean forceHttps;
    private final boolean unproxy;
    private final boolean allowRelativeUrls;
    private final Set<String> bypassSchemes;
    private final Set<String> allowedSchemes;
    private final boolean jsonMarkup;
    private final ImmutableHiderefererConfig hiderefererConfig;
    private final ImmutableUrlProcessorConfig urlProcessorConfig;
    private final boolean dropPathAndQuery;

    public ImmutableUrlSanitizingConfig(final UrlSanitizingConfig config)
        throws ConfigException
    {
        ResizerConfig resizerConfig = config.resizerConfig();
        if (resizerConfig == null) {
            this.resizerConfig = null;
        } else {
            this.resizerConfig = new ImmutableResizerConfig(resizerConfig);
        }
        forceHttps = FORCE_HTTPS.validate(config.forceHttps());
        unproxy = UNPROXY.validate(config.unproxy());
        allowRelativeUrls =
            ALLOW_RELATIVE_URLS.validate(config.allowRelativeUrls());
        dropPathAndQuery =
            DROP_PATH_AND_QUERY.validate(config.dropPathAndQuery());
        bypassSchemes =
            CollectionCompactor.compact(
                new HashSet<>(
                    BYPASS_SCHEMES.validate(config.bypassSchemes())));
        allowedSchemes =
            CollectionCompactor.compact(
                new HashSet<>(
                    ALLOWED_SCHEMES.validate(config.allowedSchemes())));
        jsonMarkup = JSON_MARKUP.validate(config.jsonMarkup());
        HiderefererConfig hiderefererConfig = config.hiderefererConfig();
        if (hiderefererConfig == null) {
            this.hiderefererConfig = null;
        } else {
            this.hiderefererConfig =
                new ImmutableHiderefererConfig(hiderefererConfig);
        }

        UrlProcessorConfigBuilder urlProcessorConfig =
            new UrlProcessorConfigBuilder();
        if (forceHttps) {
            urlProcessorConfig.defaultScheme("https");
        } else {
            urlProcessorConfig.defaultScheme("http");
        }
        urlProcessorConfig.requireSchemeForNonMailto(true);
        this.urlProcessorConfig = urlProcessorConfig.build();

        if (unproxy && resizerConfig == null) {
            throw new ConfigException(
                "Unproxy is set, but no resizer configured");
        }
    }

    @Override
    public ImmutableResizerConfig resizerConfig() {
        return resizerConfig;
    }

    @Override
    public boolean forceHttps() {
        return forceHttps;
    }

    @Override
    public boolean unproxy() {
        return unproxy;
    }

    @Override
    public boolean allowRelativeUrls() {
        return allowRelativeUrls;
    }

    @Override
    public boolean dropPathAndQuery() {
        return dropPathAndQuery;
    }

    @Override
    public Set<String> bypassSchemes() {
        return bypassSchemes;
    }

    @Override
    public Set<String> allowedSchemes() {
        return allowedSchemes;
    }

    @Override
    public boolean jsonMarkup() {
        return jsonMarkup;
    }

    @Override
    public ImmutableHiderefererConfig hiderefererConfig() {
        return hiderefererConfig;
    }

    public ImmutableUrlProcessorConfig urlProcessorConfig() {
        return urlProcessorConfig;
    }
}

