package ru.yandex.sanitizer2.config;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.regex.Pattern;

import ru.yandex.function.GenericCachingFunction;
import ru.yandex.function.GenericFunction;
import ru.yandex.parser.config.ParameterConfig;
import ru.yandex.parser.string.BooleanParser;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.sanitizer2.AttrObfuscator;

// Common config for attribute and css-properties values validation
public interface PropertyConfig {
    static Map<String, Pattern> patternsCache = new ConcurrentHashMap<>();

    ParameterConfig<Boolean> INHERITED = new ParameterConfig<>(
        "inherited",
        BooleanParser.INSTANCE,
        GenericFunction.identity());
    ParameterConfig<Boolean> TRIM = new ParameterConfig<>(
        "trim",
        BooleanParser.INSTANCE,
        GenericFunction.identity());
    ParameterConfig<Boolean> LOWERCASE = new ParameterConfig<>(
        "lowercase",
        BooleanParser.INSTANCE,
        GenericFunction.identity());
    ParameterConfig<Pattern> SPLIT_PATTERN = new ParameterConfig<>(
        "split-pattern",
        new GenericCachingFunction<>(Pattern::compile, patternsCache),
        GenericFunction.identity(),
        true);
    ParameterConfig<String> SEPARATOR = new ParameterConfig<>(
        "separator",
        GenericFunction.identity(),
        GenericFunction.identity(),
        true);
    ParameterConfig<Set<String>> PRESERVED_VALUES = new ParameterConfig<>(
        "preserved-values",
        new CollectionParser<>(NonEmptyValidator.TRIMMED, HashSet::new),
        GenericFunction.identity());
    ParameterConfig<Function<String, String>> OBFUSCATOR =
        new ParameterConfig<>(
            "obfuscator",
            new EnumParser<>(AttrObfuscator.class),
            GenericFunction.identity());
    ParameterConfig<Pattern> PATTERN = new ParameterConfig<>(
        "pattern",
        new GenericCachingFunction<>(Pattern::compile, patternsCache),
        GenericFunction.identity(),
        true);
    ParameterConfig<Pattern> SUBPATTERN = new ParameterConfig<>(
        "subpattern",
        new GenericCachingFunction<>(Pattern::compile, patternsCache),
        GenericFunction.identity(),
        true);

    // If non-null, this attribute can contain url, which should be accounted
    // in markup and (possibly) wrapped with image resizer
    UrlConfig urlConfig();

    // Indicates that this property inherited by nested elements but can be
    // overwritten without side effects
    // If value is missing, corresponding shorthand property cannot be folded
    boolean inherited();

    // Indicates that this property value must be trimmed before normalization
    boolean trim();

    // Indicates that this property value must be lowercased before
    // normalization
    boolean lowercase();

    // If not null, tokenizes value by split pattern, sanitize each part
    // separately and concatenate them all back.
    // If any parts sanitized to null, then whole property will be erased
    Pattern splitPattern();

    // If split-pattern is defined, optional separator can be used instead of
    // text matched by split pattern
    String separator();

    // These values will be preserved and won't be obfuscated or modified with
    // pattern
    Set<String> preservedValues();

    // Attributes and properties obfuscation rule
    Function<String, String> obfuscator();

    // If not null, requires value to match pattern
    Pattern pattern();

    // If not null, performs pattern replacement for matched attribute
    Map<String, String> replacement();

    // If not null, requires value to be quantity and not less than minValue
    // Matching works in two steps:
    //  1. Check if pattern is null.
    //  2. pattern is null. Property may have any value
    //  3. pattern is not null. Create matcher
    //      a. If pattern is not matched, remove property
    //      b. Iterate through replacement.* keys
    //          i) If there is named capturing group with this name, perform
    //             replacement
    //          ii) If no corresponding replacement found, leave unchanged
    //      c. Iterate through min-value.* and max-value.* keys
    //          i) If there is named capturing group with this name, parse
    //             value as double and check boundaries, discard on boundary
    //             check failure
    //          ii) If there is no such named capturing group, leave unchanged
    //
    //  For easy of use, config loading consist of the following steps:
    //  1. Check if "min-value.px" key is set
    //  2. If so, set the following default values:
    //      min-value.in  = $(min-value.px) / 96
    //      min-value.cm  = $(min-value.in) * 2.54
    //      min-value.mm  = $(min-value.cm) * 10
    //      min-value.pt  = $(min-value.in) * 72
    //      min-value.pc  = $(min-value.pt) / 12
    //      min-value.em  = $(min-value.px) / 16     <- this may be not true
    //      min-value.pct = $(min-value.em) * 100    <- this may be not true
    //      (see https://www.w3schools.com/cssref/css_units.asp for details)
    //  3. Load other units restrictions, override values calculated at step 2
    Map<String, Double> minValue();

    // If not null, requires value to be quantity and not greater than maxValue
    Map<String, Double> maxValue();

    // If not null, iterates over value with find() and perform each matcher
    // group replacement according to subreplacement map
    Pattern subpattern();

    // If subpattern not null, performs replacement for matched group
    Map<String, String> subreplacement();
}

