package ru.yandex.http.server.async;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.channels.ByteChannel;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.concurrent.Executor;

import org.apache.http.nio.reactor.IOSession;

public class AsyncCloseableIOSession extends FilterIOSession {
    private final ActiveConnection closeCallback;
    private final Selector selector;
    private final Executor executor;
    private final SocketAddress remoteAddress;
    private final SocketAddress localAddress;
    private volatile SelectionKey key = null;
    private volatile LoggingNHttpServerConnection conn = null;
    private boolean closed = false;
    private boolean closeCompleted = false;
    private boolean shutdowned = false;

    // CSOFF: ParameterNumber
    public AsyncCloseableIOSession(
        final IOSession session,
        final ActiveConnection closeCallback,
        final Selector selector,
        final Executor executor)
    {
        super(session);
        this.closeCallback = closeCallback;
        this.selector = selector;
        this.executor = executor;
        remoteAddress = session.getRemoteAddress();
        localAddress = session.getLocalAddress();
    }
    // CSON: ParameterNumber

    public void setConnection(final LoggingNHttpServerConnection conn) {
        this.conn = conn;
    }

    @Override
    public SocketAddress getRemoteAddress() {
        return remoteAddress;
    }

    @Override
    public SocketAddress getLocalAddress() {
        return localAddress;
    }

    private void beforeAsyncClose() {
        session.setEventMask(0);
        ByteChannel channel = session.channel();
        if (channel instanceof SocketChannel) {
            try {
                ((SocketChannel) channel).shutdownOutput();
            } catch (IOException e) {
                // Ignore, nothing we can do here
            }
        }
    }

    private void beforeClose() {
        ByteChannel channel = session.channel();
        if (channel instanceof SelectableChannel) {
            try {
                key = ((SelectableChannel) channel).register(selector, 0);
            } catch (ClosedChannelException e) {
                key = null;
            }
        }
    }

    private void afterClose() {
        LoggingNHttpServerConnection conn = this.conn;
        if (conn != null) {
            try {
                conn.close();
            } catch (IOException e) {
                // Ignore, nothing we can do here
            }
        }
        SelectionKey key = this.key;
        if (key != null) {
            key.cancel();
            try {
                if (selector.selectNow() > 0) {
                    synchronized (selector) {
                        selector.selectedKeys().clear();
                    }
                }
            } catch (IOException e) {
                // Ignore, nothing we can do here
            }
        }
    }

    @Override
    public void close() {
        synchronized (this) {
            if (closed) {
                return;
            }
            closed = true;
        }
        closeCallback.markClosed();
        beforeAsyncClose();
        executor.execute(
            new Runnable() {
                @Override
                public void run() {
                    beforeClose();
                    AsyncCloseableIOSession.super.close();
                    afterClose();
                    synchronized (AsyncCloseableIOSession.this) {
                        closeCompleted = true;
                        if (!shutdowned) {
                            return;
                        }
                    }
                    AsyncCloseableIOSession.super.shutdown();
                }
            });
    }

    @Override
    public void shutdown() {
        boolean closed;
        boolean closeCompleted;
        synchronized (this) {
            if (shutdowned) {
                return;
            }
            shutdowned = true;
            closed = this.closed;
            this.closed = true;
            closeCompleted = this.closeCompleted;
        }
        closeCallback.markShutdowned();
        if (closed) {
            if (closeCompleted) {
                executor.execute(
                    new Runnable() {
                        @Override
                        public void run() {
                            AsyncCloseableIOSession.super.shutdown();
                        }
                    });
            }
        } else {
            beforeAsyncClose();
            executor.execute(
                new Runnable() {
                    @Override
                    public void run() {
                        beforeClose();
                        AsyncCloseableIOSession.super.shutdown();
                        afterClose();
                    }
                });
        }
    }
}

