package ru.yandex.http.server.async;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.InetSocketAddress;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLSession;

import org.apache.http.config.ConnectionConfig;
import org.apache.http.impl.nio.DefaultHttpServerIODispatch;
import org.apache.http.nio.NHttpConnectionFactory;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.nio.protocol.HttpAsyncResponseProducer;
import org.apache.http.nio.reactor.IOSession;
import org.apache.http.nio.reactor.ListenerEndpoint;
import org.apache.http.nio.reactor.ssl.SSLSetupHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.ExecutorServiceCloser;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.SingleNamedThreadFactory;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.nio.LoggingIOReactorExceptionHandler;
import ru.yandex.http.util.nio.ReactorUtils;
import ru.yandex.http.util.server.AbstractHttpServer;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.stater.ImmutableGolovanAlertsConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.stater.ThreadPoolStater;

public class BaseAsyncServer<T extends ImmutableBaseServerConfig>
    extends AbstractHttpServer<T, HttpAsyncRequestHandler<?>>
    implements SSLSetupHandler
{
    protected final ThreadPoolExecutor executor;
    protected final ThreadPoolExecutor statExecutor;

    private final LimitingListeningIOReactor reactor;
    private final BaseAsyncService service;
    private volatile boolean bound = false;
    private ListenerEndpoint endpoint;
    private InetSocketAddress address;
    private ListenerEndpoint httpEndpoint;
    private InetSocketAddress httpAddress;

    public BaseAsyncServer(final T config) throws IOException {
        this(config, new CompositeHttpAsyncRequestHandlerMapper());
    }

    public BaseAsyncServer(
        final T config,
        final CompositeHttpAsyncRequestHandlerMapper handlerMapper)
        throws IOException
    {
        super(config, handlerMapper);
        executor = new ThreadPoolExecutor(
            config.workers(),
            config.workers(),
            1,
            TimeUnit.MINUTES,
            new LinkedBlockingQueue<>(),
            new NamedThreadFactory(
                new ThreadGroup(
                    getThreadGroup(),
                    config.name() + "-AsyncExecutor"),
                true),
            new ThreadPoolExecutor.DiscardPolicy());
        closeChain.add(new ExecutorServiceCloser(executor));

        statExecutor = new ThreadPoolExecutor(
            1,
            1,
            1,
            TimeUnit.MINUTES,
            new LinkedBlockingQueue<>(),
            new SingleNamedThreadFactory(
                getThreadGroup(),
                config.name() + "-StatExecutor",
                true),
            new ThreadPoolExecutor.CallerRunsPolicy());
        closeChain.add(new ExecutorServiceCloser(statExecutor));

        reactor = new LimitingListeningIOReactor(
            config,
            new NamedThreadFactory(getThreadGroup(), config.name() + '-'),
            this);
        closeChain.add(new ReactorCloser(reactor));
        reactor.setExceptionHandler(
            new LoggingIOReactorExceptionHandler(logger));
        if (!config.lazyBind()) {
            bind();
            bound = true;
        }
        service = new BaseAsyncService(
            handlerMapper,
            serviceContextRenewalTask,
            this,
            () -> pingEnabled(null),
            logger);

        register(
            new Pattern<>("/ping", false),
            new PingHandler(this),
            false);
        register(
            new Pattern<>("/disablePing", false),
            new DisablePingHandler(this::disablePing),
            false);
        register(
            new Pattern<>("/disable-ping", false),
            new DisablePingHandler(this::disablePing),
            false);
        register(
            new Pattern<>("/enable-ping", false),
            new EnablePingHandler(this::enablePing),
            false);
        register(
            new Pattern<>("/config-update", false),
            new DynamycConfigUpdateHandler(this),
            false);
        register(
            new Pattern<>("/logrotate", false),
            new DelegatedHttpAsyncRequestHandler<>(
                new LogRotateHandler(this),
                this),
            false);
        register(new Pattern<>("/status", false), new StatusHandler(this), false);
        register(
            new Pattern<>("/stat", false),
            new DelegatedHttpAsyncRequestHandler<>(
                new StatHandler(this),
                this,
                statExecutor),
            false);
        register(new Pattern<>("/systemexit", false), new SystemExitHandler(), false);
        register(
            new Pattern<>("/force-gc", false),
            new DelegatedHttpAsyncRequestHandler<>(
                new ForceGcHandler(),
                this),
            false);

        register(
            new Pattern<>("/add-debug-flag", false),
            new AddDebugFlagHandler(this),
            false);
        register(
            new Pattern<>("/remove-debug-flag", false),
            new RemoveDebugFlagHandler(this),
            false);
        register(
            new Pattern<>("/debug-flags", false),
            new DebugFlagsHandler(this),
            false);

        if (ibmDumper != null) {
            DumpHandler dumpHandler = new DumpHandler(ibmDumper);
            register(new Pattern<>("/javadump", false), dumpHandler, false);
            register(new Pattern<>("/systemdump", false), dumpHandler, false);
            register(new Pattern<>("/heapdump", false), dumpHandler, false);
        }

        register(
            new Pattern<>("/custom-golovan-panel", false),
            new CustomGolovanPanelHandler(this),
            false);
        register(
            new Pattern<>("/custom-alerts-config", false),
            new CustomAlertsConfigHandler(this),
            false);
        if (golovanPanelConfig != null) {
            register(
                new Pattern<>("/generate-golovan-panel", false),
                new GenerateGolovanPanelHandler(this),
                false);
            ImmutableGolovanAlertsConfig alerts = golovanPanelConfig.alerts();
            if (alerts != null) {
                register(
                    new Pattern<>("/generate-alerts-config", false),
                    new GenerateAlertsConfigHandler(this),
                    false);
            }
        }

        registerStater(new BaseAsyncServerStater(executor, statExecutor));
        registerStater(new ThreadPoolStater(executor, "synchronous-ops-"));
    }

    private void bind() throws IOException {
        endpoint = reactor.bind(new InetSocketAddress(config.port()));
        closeChain.add(new EndpointCloser(endpoint));
        address = (InetSocketAddress) endpoint.getAddress();
        if (httpsConfig != null && httpsConfig.httpPort() >= 0) {
            httpEndpoint =
                reactor.bind(new InetSocketAddress(httpsConfig.httpPort()));
            closeChain.add(new EndpointCloser(httpEndpoint));
            httpAddress = (InetSocketAddress) httpEndpoint.getAddress();
        } else {
            httpEndpoint = endpoint;
            httpAddress = address;
        }
    }

    private void ensureBound() throws IOException {
        if (!bound) {
            synchronized (this) {
                if (!bound) {
                    beforeLazyBind();
                    bind();
                    bound = true;
                }
            }
        }
    }

    /**
     * Called before actual binding only if lazy bind enabled
     */
    protected void beforeLazyBind() {
    }

    @Override
    protected HttpAsyncRequestHandler<?> wrap(
        final HttpAsyncRequestHandler<?> handler)
    {
        return new OverridableHttpAsyncRequestHandler(handler);
    }

    @Override
    protected HttpAsyncRequestHandler<?> unwrap(
        final HttpAsyncRequestHandler<?> handler)
    {
        if (handler instanceof OverridableHttpAsyncRequestHandler) {
            return ((OverridableHttpAsyncRequestHandler) handler).handler();
        } else {
            return handler;
        }
    }

    @Override
    public void start() throws IOException {
        executor.prestartAllCoreThreads();
        statExecutor.prestartAllCoreThreads();
        super.start();
    }

    @Override
    public InetSocketAddress address() throws IOException {
        ensureBound();
        return address;
    }

    @Override
    public InetSocketAddress httpAddress() throws IOException {
        ensureBound();
        return httpAddress;
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put("synchronous_tasks", executor.getQueue().size());
        status.put("synchronous_tasks_workers", executor.getActiveCount());
        status.put("stat_tasks", statExecutor.getQueue().size());
        ReactorUtils.status(status, reactor);
        return status;
    }

    public HttpAsyncResponseProducer handleException(
        final Exception e,
        final HttpContext context)
    {
        return service.handleException(e, context);
    }

    @Override
    public void run() {
        try {
            ConnectionConfig connectionConfig = config.toConnectionConfig();
            NHttpConnectionFactory<LoggingNHttpServerConnection> connFactory;
            if (httpsConfig == null) {
                connFactory = new LoggingNHttpServerConnectionFactory(
                    connectionConfig,
                    executor);
            } else {
                int httpPort = httpsConfig.httpPort();
                if (httpPort >= 0) {
                    connFactory = new ConditionalNHttpServerConnectionFactory(
                        httpPort(),
                        new LoggingNHttpServerConnectionFactory(
                            connectionConfig,
                            executor),
                        new SSLLoggingNHttpServerConnectionFactory(
                            connectionConfig,
                            executor,
                            sslContext,
                            this));
                } else {
                    connFactory = new SSLLoggingNHttpServerConnectionFactory(
                        connectionConfig,
                        executor,
                        sslContext,
                        this);
                }
            }
            reactor.execute(
                new DefaultHttpServerIODispatch<>(service, connFactory));
        } catch (InterruptedIOException e) {
            logger.info("Server interrupted");
        } catch (IOException e) {
            logger.log(Level.SEVERE, "Reactor error", e);
        }
    }

    // SSLSetupHandler implementation
    @Override
    public void initalize(final SSLEngine sslEngine) {
        config.httpsConfig().initialize(sslEngine);
    }

    @Override
    public void verify(
        final IOSession ioSession,
        final SSLSession sslSession)
    {
    }

    private static class BaseAsyncServerStater implements Stater {
        private final ThreadPoolExecutor executor;
        private final ThreadPoolExecutor statExecutor;

        BaseAsyncServerStater(
            final ThreadPoolExecutor executor,
            final ThreadPoolExecutor statExecutor)
        {
            this.executor = executor;
            this.statExecutor = statExecutor;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            int synchronousTasks = executor.getQueue().size();
            statsConsumer.stat("synchronous-tasks_axxx", synchronousTasks);
            statsConsumer.stat("synchronous-tasks_ammx", synchronousTasks);
            statsConsumer.stat("synchronous-tasks_ammm", synchronousTasks);

            int synchronousTasksWorkers = executor.getActiveCount();
            statsConsumer.stat(
                "synchronous-tasks-workers_axxx",
                synchronousTasksWorkers);
            statsConsumer.stat(
                "synchronous-tasks-workers_ammx",
                synchronousTasksWorkers);
            statsConsumer.stat(
                "synchronous-tasks-workers_ammm",
                synchronousTasksWorkers);

            int statTasks = statExecutor.getQueue().size();
            statsConsumer.stat("stat-tasks_axxx", statTasks);
            statsConsumer.stat("stat-tasks_ammx", statTasks);
            statsConsumer.stat("stat-tasks_ammm", statTasks);
        }
    }

    private static class ReactorCloser
        implements GenericAutoCloseable<IOException>
    {
        private final LimitingListeningIOReactor reactor;

        ReactorCloser(final LimitingListeningIOReactor reactor) {
            this.reactor = reactor;
        }

        @Override
        public void close() throws IOException {
            reactor.shutdown();
        }
    }

    private static class EndpointCloser
        implements GenericAutoCloseable<IOException>
    {
        private final ListenerEndpoint endpoint;

        EndpointCloser(final ListenerEndpoint endpoint) {
            this.endpoint = endpoint;
        }

        @Override
        public void close() {
            endpoint.close();
        }
    }
}

