package ru.yandex.http.server.async;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.BooleanSupplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.ConnectionClosedException;
import org.apache.http.Header;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.HttpResponseFactory;
import org.apache.http.HttpResponseInterceptor;
import org.apache.http.HttpVersion;
import org.apache.http.concurrent.Cancellable;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.DefaultHttpResponseFactory;
import org.apache.http.nio.NHttpServerConnection;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncResponseProducer;
import org.apache.http.nio.protocol.HttpAsyncService;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;
import org.apache.http.protocol.ImmutableHttpProcessor;
import org.apache.http.protocol.ResponseServer;

import ru.yandex.client.tvm2.Tvm2ServiceContextRenewalTask;
import ru.yandex.http.util.BasicExceptionLogger;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.HeadfulResponseContent;
import ru.yandex.http.util.IOHttpException;
import ru.yandex.http.util.ResponseConnControl;
import ru.yandex.http.util.ResponseXRequestId;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.StaticHeaders;
import ru.yandex.http.util.YandexReasonPhraseCatalog;
import ru.yandex.http.util.nio.BasicAsyncResponseProducer;
import ru.yandex.http.util.server.BaseServerDynamicConfig;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.http.util.server.ResponseHostname;
import ru.yandex.http.util.server.ServerConfigProvider;
import ru.yandex.http.util.server.SessionContext;
import ru.yandex.http.util.server.SessionId;

public class BaseAsyncService extends HttpAsyncService {
    private static final HttpResponseFactory RESPONSE_FACTORY =
        new DefaultHttpResponseFactory(YandexReasonPhraseCatalog.INSTANCE);
    private static final Header CONN_CLOSE_HEADER =
        HeaderUtils.createHeader(HTTP.CONN_DIRECTIVE, HTTP.CONN_CLOSE);

    private final SessionContext sessionContextInterceptor;
    private final Logger logger;

    // CSOFF: ParameterNumber
    public BaseAsyncService(
        final CompositeHttpAsyncRequestHandlerMapper requestMapper,
        final Tvm2ServiceContextRenewalTask serviceContextRenewalTask,
        final ServerConfigProvider<ImmutableBaseServerConfig, BaseServerDynamicConfig> config,
        final BooleanSupplier keepAlive,
        final Logger logger)
    {
        this(
            requestMapper,
            new SessionContext(
                requestMapper,
                serviceContextRenewalTask,
                config),
            config.staticConfig(),
            keepAlive,
            logger);
    }

    public BaseAsyncService(
        final CompositeHttpAsyncRequestHandlerMapper requestMapper,
        final SessionContext sessionContextInterceptor,
        final ImmutableBaseServerConfig config,
        final BooleanSupplier keepAlive,
        final Logger logger)
    {
        super(
            new ImmutableHttpProcessor(
                new HttpRequestInterceptor[] {sessionContextInterceptor},
                createResponseInterceptors(config, keepAlive)),
            new LoggingConnectionReuseStrategy(logger),
            RESPONSE_FACTORY,
            requestMapper,
            null,
            new BasicExceptionLogger(
                logger,
                "Exception in async service"));
        this.sessionContextInterceptor = sessionContextInterceptor;
        this.logger = logger;
    }
    // CSON: ParameterNumber

    private static HttpResponseInterceptor[] createResponseInterceptors(
        final ImmutableBaseServerConfig config,
        final BooleanSupplier keepAlive)
    {
        List<HttpResponseInterceptor> responseInterceptors = new ArrayList<>();
        List<Header> staticHeaders = config.staticHeaders();
        if (!staticHeaders.isEmpty()) {
            responseInterceptors.add(
                new StaticHeaders(
                    staticHeaders.toArray(new Header[staticHeaders.size()])));
        }
        responseInterceptors.add(
            new ResponseConnControl(
                keepAlive,
                config.timeout(),
                config.keepAliveStatuses()));
        responseInterceptors.add(
            new NResponseContentEncoding(config.gzip(), config.bufferSize()));
        responseInterceptors.add(HeadfulResponseContent.INSTANCE);
        responseInterceptors.add(ResponseXRequestId.INSTANCE);
        if (!config.briefHeaders()) {
            responseInterceptors.add(SessionId.INSTANCE);
            responseInterceptors.add(ResponseHostname.INSTANCE);
            responseInterceptors.add(new ResponseServer(config.origin()));
        }
        return responseInterceptors.toArray(
            new HttpResponseInterceptor[responseInterceptors.size()]);
    }

    public static HttpResponse exceptionToResponse(
        final ServerException e,
        final HttpContext context)
    {
        HttpResponse response = RESPONSE_FACTORY.newHttpResponse(
            HttpVersion.HTTP_1_1,
            e.statusCode(),
            context);
        response.addHeader(CONN_CLOSE_HEADER);
        response.setEntity(
            new NStringEntity(
                e.toStackTrace(),
                ContentType.TEXT_PLAIN
                    .withCharset(StandardCharsets.UTF_8)));
        return response;
    }

    // XXX: We don't use standard .exception(...) function for response
    // submission because it's too hard to synchronize response submission
    // using .exception(...)
    @Override
    public HttpAsyncResponseProducer handleException(
        final Exception e,
        final HttpContext context)
    {
        HttpCoreContext coreContext = HttpCoreContext.adapt(context);
        Logger logger =
            coreContext.getAttribute(BaseAsyncServer.LOGGER, Logger.class);
        if (logger == null) {
            logger = this.logger;
        }
        String connection = Objects.toString(coreContext.getConnection());
        StringBuilder sb = sessionContextInterceptor.requestToStringBuilder(
            "Exception occured while processing ",
            coreContext.getRequest(),
            2 + 2 + connection.length());
        sb.append(" on ");
        sb.append(connection);
        logger.log(Level.WARNING, new String(sb), e);
        if (e instanceof ServerException) {
            return new BasicAsyncResponseProducer(
                exceptionToResponse((ServerException) e, context));
        } else {
            return super.handleException(e, context);
        }
    }

    @Override
    protected void handleAlreadySubmittedResponse(
        final Cancellable callback,
        final HttpContext context)
    {
        StringBuilder sb = new StringBuilder(
            "Response already submitted, cancelling callback: ");
        sb.append(callback);
        for (StackTraceElement frame: Thread.currentThread().getStackTrace()) {
            sb.append("\n\tat ");
            sb.append(frame);
        }
        Logger logger = (Logger) context.getAttribute(BaseAsyncServer.LOGGER);
        logger.warning(new String(sb));
        callback.cancel();
    }

    @Override
    protected void handleAlreadySubmittedResponse(
        final HttpAsyncResponseProducer responseProducer,
        final HttpContext context)
    {
        StringBuilder sb =
            new StringBuilder("Suppressing response submission: ");
        sb.append(responseProducer);
        for (StackTraceElement frame: Thread.currentThread().getStackTrace()) {
            sb.append("\n\tat ");
            sb.append(frame);
        }
        Logger logger = (Logger) context.getAttribute(BaseAsyncServer.LOGGER);
        logger.warning(new String(sb));
        try {
            responseProducer.close();
        } catch (IOException e) {
            logger.log(Level.WARNING, "Failed to close response producer", e);
        }
    }

    private static void shutdownQuietly(final NHttpServerConnection conn) {
        try {
            conn.shutdown();
        } catch (IOException e) {
        }
    }

    @Override
    public void exception(
        final NHttpServerConnection conn,
        final Exception exception)
    {
        if (exception instanceof ConnectionClosedException) {
            logger.info("Client closed connection: " + conn);
            shutdownQuietly(conn);
        } else if (exception instanceof IOHttpException) {
            logger.log(
                Level.WARNING,
                "IOHttpException occured on " + conn,
                exception);
            super.exception(conn, ((IOHttpException) exception).getCause());
        } else if (exception instanceof IOException) {
            if ("Connection reset by peer".equals(exception.getMessage())) {
                logger.info("Connection reset by peer: " + conn);
            } else {
                logger.log(
                    Level.WARNING,
                    "I/O error occured on " + conn,
                    exception);
            }
            shutdownQuietly(conn);
        } else {
            logger.log(
                Level.WARNING,
                "Exception occured on " + conn,
                exception);
            super.exception(conn, exception);
        }
    }

    @Override
    public void timeout(final NHttpServerConnection conn) {
        logger.warning("Client connection timeout occured on " + conn);
        shutdownQuietly(conn);
    }

    @Override
    public void closed(final NHttpServerConnection conn) {
        super.closed(conn);
        try {
            conn.close();
        } catch (IOException e) {
        }
    }
}

