package ru.yandex.http.server.async;

import java.io.IOException;
import java.util.concurrent.CancellationException;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.Cancellable;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.util.ServiceUnavailableException;

public class DelegatedHttpAsyncRequestHandler<T>
    implements HttpAsyncRequestHandler<T>
{
    private final HttpAsyncRequestHandler<T> handler;
    private final BaseAsyncServer<?> server;
    private final Executor executor;

    public DelegatedHttpAsyncRequestHandler(
        final HttpAsyncRequestHandler<T> handler,
        final BaseAsyncServer<?> server)
    {
        this(handler, server, server.executor);
    }

    public DelegatedHttpAsyncRequestHandler(
        final HttpAsyncRequestHandler<T> handler,
        final BaseAsyncServer<?> server,
        final Executor executor)
    {
        this.handler = handler;
        this.server = server;
        this.executor = executor;
    }

    @Override
    public HttpAsyncRequestConsumer<T> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        return handler.processRequest(request, context);
    }

    @Override
    public void handle(
        final T request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
    {
        try {
            executor.execute(new DelegatedTask(request, exchange, context));
        } catch (RuntimeException e) {
            exchange.submitResponse(server.handleException(e, context));
        }
    }

    @Override
    public String toString() {
        return handler.toString();
    }

    private class DelegatedTask
        extends AtomicBoolean
        implements Cancellable, Runnable
    {
        private static final long serialVersionUID = 0L;

        private final T request;
        private final HttpAsyncExchange exchange;
        private final HttpContext context;

        DelegatedTask(
            final T request,
            final HttpAsyncExchange exchange,
            final HttpContext context)
        {
            this.request = request;
            this.exchange = exchange;
            this.context = context;
            exchange.setCallback(this);
        }

        @Override
        public boolean cancel() {
            return compareAndSet(false, true);
        }

        @Override
        public void run() {
            try {
                if (get()) {
                    throw new CancellationException("Request cancelled");
                }
                handler.handle(request, exchange, context);
            } catch (Exception e) {
                exchange.submitResponse(server.handleException(e, context));
            } catch (Error e) {
                exchange.submitResponse(
                    server.handleException(
                        new ServiceUnavailableException(e),
                        context));
            }
        }
    }
}

