package ru.yandex.http.server.async;

import java.io.IOException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.impl.ConnSupport;
import org.apache.http.impl.entity.StrictContentLengthStrategy;
import org.apache.http.impl.io.HttpTransportMetricsImpl;
import org.apache.http.impl.nio.DefaultNHttpServerConnection;
import org.apache.http.nio.ContentDecoder;
import org.apache.http.nio.ContentEncoder;
import org.apache.http.nio.NHttpServerEventHandler;
import org.apache.http.nio.reactor.SessionInputBuffer;
import org.apache.http.nio.reactor.SessionOutputBuffer;
import org.apache.http.protocol.HttpContext;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.ContentDecoderListener;
import ru.yandex.http.util.nio.ContentEncoderListener;
import ru.yandex.http.util.nio.CountingContentDecoder;
import ru.yandex.http.util.nio.CountingContentEncoder;
import ru.yandex.http.util.server.HttpSessionInfo;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.stater.RequestInfo;
import ru.yandex.util.timesource.TimeSource;

public class LoggingNHttpServerConnection
    extends DefaultNHttpServerConnection
    implements ContentDecoderListener,
        ContentEncoderListener,
        LoggingServerConnection
{
    private final int timeout;
    private volatile long start = TimeSource.INSTANCE.currentTimeMillis();
    private volatile boolean firstRequest = true;
    private Logger accessLogger = null;
    private Consumer<RequestInfo> stater = null;
    private HttpSessionInfo sessionInfo = null;
    private GenericAutoCloseable<RuntimeException> resourcesReleaser = null;
    private long sessionStart = 0L;
    private long processingStart = 0L;
    private long bytesRead = -1L;
    private long bytesWritten = -1L;

    public LoggingNHttpServerConnection(
        final AsyncCloseableIOSession session,
        final ConnectionConfig config)
    {
        super(
            session,
            config.getBufferSize(),
            config.getFragmentSizeHint(),
            null,
            ConnSupport.createDecoder(config),
            ConnSupport.createEncoder(config),
            config.getMessageConstraints(),
            StrictContentLengthStrategy.INSTANCE,
            StrictContentLengthStrategy.INSTANCE,
            null,
            null);

        session.setConnection(this);
        timeout = getSocketTimeout();
    }

    //CSOFF: ParameterNumber
    @Override
    public void setRequestContext(
        final Logger accessLogger,
        final Consumer<RequestInfo> stater,
        final GenericAutoCloseable<RuntimeException> resourcesReleaser,
        final HttpContext context)
    {
        synchronized (this) {
            this.accessLogger = accessLogger;
            this.stater = stater;
            this.resourcesReleaser = resourcesReleaser;
            sessionInfo = new HttpSessionInfo(context);
        }
        sessionStart = start;
        processingStart = TimeSource.INSTANCE.currentTimeMillis();
        bytesWritten = -1L;
    }
    //CSON: ParameterNumber

    @Override
    public synchronized void setStater(final Consumer<RequestInfo> stater) {
        this.stater = stater;
    }

    @Override
    public synchronized void setSessionInfo(
        final String name,
        final Object value)
    {
        if (sessionInfo != null) {
            sessionInfo.put(name, value);
        }
    }

    @Override
    public long requestStartTime() {
        return sessionStart;
    }

    @Override
    public void incrementRead(final long read) {
        bytesRead += read;
    }

    @Override
    public void onDecoderIOException(final IOException e) {
        log(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
    }

    @Override
    public void onDecoderError(final Throwable t) {
        log(YandexHttpStatus.SC_ENTITY_READ_FAILED);
    }

    @Override
    public void incrementWritten(final long written) {
        bytesWritten += written;
    }

    @Override
    public void onEncoderIOException(final IOException e) {
        log(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
    }

    @Override
    public void onEncoderError(final Throwable t) {
        log(YandexHttpStatus.SC_ENTITY_SEND_FAILED);
    }

    public void log(final int statusCode) {
        Logger accessLogger;
        HttpSessionInfo sessionInfo;
        Consumer<RequestInfo> stater;
        GenericAutoCloseable<RuntimeException> resourcesReleaser;
        synchronized (this) {
            sessionInfo = this.sessionInfo;
            this.sessionInfo = null;
            accessLogger = this.accessLogger;
            this.accessLogger = null;
            stater = this.stater;
            this.stater = null;
            resourcesReleaser = this.resourcesReleaser;
            this.resourcesReleaser = null;
        }
        if (sessionInfo == null) {
            return;
        }
        if (resourcesReleaser != null) {
            resourcesReleaser.close();
        }
        sessionInfo.put(STATUS, Integer.toString(statusCode));
        long requestLength;
        if (bytesRead == -1L) {
            requestLength = 0L;
        } else {
            requestLength = bytesRead;
            sessionInfo.put(REQUEST_LENGTH, Long.toString(bytesRead));
        }

        long responseLength;
        if (bytesWritten == -1L) {
            responseLength = 0L;
        } else {
            responseLength = bytesWritten;
            sessionInfo.put(RESPONSE_LENGTH, Long.toString(bytesWritten));
        }

        RequestInfo info = new RequestInfo(
            TimeSource.INSTANCE.currentTimeMillis(),
            statusCode,
            sessionStart,
            processingStart,
            requestLength,
            responseLength);
        sessionInfo.put(REQUEST_TIME, Long.toString(info.requestTime()));
        sessionInfo.put(PROCESSING_TIME, Long.toString(info.processingTime()));
        sessionInfo.put(SESSION_START_TS, Long.toString(sessionStart));
        sessionInfo.put(PROCESS_START_TS, Long.toString(processingStart));
        accessLogger.log(Level.INFO, "", sessionInfo);
        if (stater != null) {
            stater.accept(info);
        }
    }

    @Override
    protected ContentDecoder createContentDecoder(
        final long len,
        final ReadableByteChannel channel,
        final SessionInputBuffer buffer,
        final HttpTransportMetricsImpl metrics)
    {
        bytesRead = 0L;
        ContentDecoder decoder =
            super.createContentDecoder(len, channel, buffer, metrics);
        return CountingContentDecoder.wrap(decoder, this);
    }

    @Override
    protected ContentEncoder createContentEncoder(
        final long len,
        final WritableByteChannel channel,
        final SessionOutputBuffer buffer,
        final HttpTransportMetricsImpl metrics)
    {
        bytesWritten = 0L;
        ContentEncoder encoder =
            super.createContentEncoder(len, channel, buffer, metrics);
        return CountingContentEncoder.wrap(encoder, this);
    }

    @Override
    public void consumeInput(final NHttpServerEventHandler handler) {
        if (request == null) {
            if (firstRequest) {
                firstRequest = false;
            } else if (start == 0L) {
                start = TimeSource.INSTANCE.currentTimeMillis();
            }
        }
        super.consumeInput(handler);
    }

    @Override
    public void resetInput() {
        if (request != null) {
            setSocketTimeout(0);
        }
        start = 0L;
        super.resetInput();
    }

    @Override
    public void submitResponse(final HttpResponse response)
        throws IOException, HttpException
    {
        super.submitResponse(response);
        int status = response.getStatusLine().getStatusCode();
        if (status >= HttpStatus.SC_OK) {
            setSocketTimeout(timeout);
        }
    }

    @Override
    public void shutdown() throws IOException {
        log(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
        super.shutdown();
        outbuf.reset(null);
    }

    @Override
    public void close() throws IOException {
        log(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
        super.close();
    }
}

