package ru.yandex.http.server.async;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.nio.protocol.HttpAsyncResponseProducer;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.util.nio.BasicAsyncResponseProducer;
import ru.yandex.http.util.nio.FilterHttpAsyncExchange;
import ru.yandex.http.util.server.HttpServer;

public class OverridableHttpAsyncRequestHandler
    implements HttpAsyncRequestHandler<Object>
{
    private final HttpAsyncRequestHandler<Object> handler;

    @SuppressWarnings("unchecked")
    public OverridableHttpAsyncRequestHandler(
        final HttpAsyncRequestHandler<?> handler)
    {
        this.handler = (HttpAsyncRequestHandler<Object>) handler;
    }

    public HttpAsyncRequestHandler<Object> handler() {
        return handler;
    }

    @SuppressWarnings("unchecked")
    private HttpAsyncRequestHandler<Object> selectHandler(
        final HttpContext context)
    {
        Object overriddenHandler =
            context.getAttribute(HttpServer.OVERRIDDEN_HANDLER);
        if (overriddenHandler == null) {
            return handler;
        } else {
            return (HttpAsyncRequestHandler<Object>) overriddenHandler;
        }
    }

    @Override
    public HttpAsyncRequestConsumer<Object> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        return selectHandler(context).processRequest(request, context);
    }

    @Override
    public void handle(
        final Object request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        Logger logger = (Logger) context.getAttribute(HttpServer.LOGGER);
        if (logger.isLoggable(Level.FINER)
            && exchange.getRequest() instanceof HttpEntityEnclosingRequest)
        {
            logger.finer("Handling request");
        }
        selectHandler(context).handle(
            request,
            // org.apache.http.nio.protocol.BasicAsyncResponseProducer doesn't
            // allow to override entity in response interceptor, so we use
            // ru.yandex.http.util.nio.BasicAsyncResponseProducer instead
            new WrappingHttpAsyncExchange(exchange, logger),
            context);
    }

    @Override
    public String toString() {
        return handler.toString();
    }

    private static class WrappingHttpAsyncExchange
        extends FilterHttpAsyncExchange
    {
        private final Logger logger;

        WrappingHttpAsyncExchange(
            final HttpAsyncExchange exchange,
            final Logger logger)
        {
            super(exchange);
            this.logger = logger;
        }

        private void onResponseSubmitted() {
            if (logger.isLoggable(Level.FINER)) {
                logger.finer("Response submitted");
            }
        }

        @Override
        public void submitResponse() {
            submitResponse(new BasicAsyncResponseProducer(getResponse()));
        }

        @Override
        public void submitResponse(final HttpAsyncResponseProducer producer) {
            super.submitResponse(producer);
            onResponseSubmitted();
        }
    }
}

