package ru.yandex.http.config;

import java.nio.charset.Charset;

import ru.yandex.parser.config.ConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.IntegerMemorySizeParser;

public abstract class AbstractHttpConnectionConfigBuilder
    <T extends AbstractHttpConnectionConfigBuilder<T>>
    implements ConfigBuilder<T>, HttpConnectionConfig
{
    private int bufferSize;
    private int fragmentSizeHint;
    private Charset headersCharset;
    private boolean briefHeaders;

    protected AbstractHttpConnectionConfigBuilder(
        final HttpConnectionConfig config)
    {
        bufferSize = config.bufferSize();
        fragmentSizeHint = config.fragmentSizeHint();
        headersCharset = config.headersCharset();
        briefHeaders = config.briefHeaders();
    }

    protected AbstractHttpConnectionConfigBuilder(
        final IniConfig config,
        final HttpConnectionConfig defaults)
        throws ConfigException
    {
        bufferSize = config.get(
            "buffer-size",
            defaults.bufferSize(),
            IntegerMemorySizeParser.POSITIVE);
        fragmentSizeHint =
            config.getInt("fragment-size-hint", defaults.fragmentSizeHint());
        headersCharset =
            config.getCharset("headers-charset", defaults.headersCharset());
        briefHeaders =
            config.getBoolean("brief-headers", defaults.briefHeaders());
    }

    @Override
    public int bufferSize() {
        return bufferSize;
    }

    public T bufferSize(final int bufferSize) {
        this.bufferSize = bufferSize;
        return self();
    }

    @Override
    public int fragmentSizeHint() {
        return fragmentSizeHint;
    }

    public T fragmentSizeHint(final int fragmentSizeHint) {
        this.fragmentSizeHint = fragmentSizeHint;
        return self();
    }

    @Override
    public Charset headersCharset() {
        return headersCharset;
    }

    public T headersCharset(final Charset headersCharset) {
        this.headersCharset = headersCharset;
        return self();
    }

    public void copyTo(final AbstractHttpConnectionConfigBuilder<?> other) {
        other
            .bufferSize(bufferSize)
            .fragmentSizeHint(fragmentSizeHint)
            .headersCharset(headersCharset)
            .briefHeaders(briefHeaders);
    }

    @Override
    public boolean briefHeaders() {
        return briefHeaders;
    }

    public T briefHeaders(final boolean briefHeaders) {
        this.briefHeaders = briefHeaders;
        return self();
    }
}

