package ru.yandex.http.config;

import java.nio.charset.Charset;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import org.apache.http.HttpHost;

import ru.yandex.http.util.HttpHostParser;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.stater.StatersConfig;
import ru.yandex.stater.StatersConfigBuilder;

public abstract class AbstractHttpTargetConfigBuilder
    <T extends AbstractHttpTargetConfigBuilder<T>>
    extends AbstractHttpConnectionConfigBuilder<T>
    implements HttpTargetConfig
{
    private static final String RETRIES = "retries";
    private static final String RETRY_INTERVAL = "retry-interval";
    private static final String TIMEOUT = "timeout";

    private HttpHost proxy;
    private Charset requestCharset;
    private Charset responseCharset;
    private int connections;
    private int timeout;
    private int connectTimeout;
    private int poolTimeout;
    private int sessionTimeout;
    private RetriesConfigBuilder ioRetries;
    private RetriesConfigBuilder httpRetries;
    private boolean keepAlive;
    private boolean contentCompression;
    private boolean redirects;
    private boolean redirectPosts;
    private boolean relativeRedirects;
    private boolean circularRedirects;
    private int maxRedirects;
    private boolean passReferer;
    private ClientHttpsConfigBuilder httpsConfig;
    private List<Map.Entry<String, String>> tvm2Headers;
    private StatersConfigBuilder statersConfig;

    protected AbstractHttpTargetConfigBuilder(final HttpTargetConfig config) {
        super(config);
        proxy(config.proxy());
        requestCharset(config.requestCharset());
        responseCharset(config.responseCharset());
        connections(config.connections());
        timeout(config.timeout());
        connectTimeout(config.connectTimeout());
        poolTimeout(config.poolTimeout());
        sessionTimeout(config.sessionTimeout());
        ioRetries(new RetriesConfigBuilder(config.ioRetries()));
        httpRetries(new RetriesConfigBuilder(config.httpRetries()));
        keepAlive(config.keepAlive());
        contentCompression(config.contentCompression());
        redirects(config.redirects());
        redirectPosts(config.redirectPosts());
        relativeRedirects(config.relativeRedirects());
        circularRedirects(config.circularRedirects());
        maxRedirects(config.maxRedirects());
        passReferer(config.passReferer());
        httpsConfig(config.httpsConfig());
        tvm2Headers(config.tvm2Headers());
        statersConfig(config.statersConfig());
    }

    @SuppressWarnings("StringSplitter")
    protected AbstractHttpTargetConfigBuilder(
        final IniConfig config,
        final HttpTargetConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        proxy = config.get("proxy", defaults.proxy(), HttpHostParser.INSTANCE);
        requestCharset =
            config.getCharset("request-charset", defaults.requestCharset());
        responseCharset =
            config.getCharset("response-charset", defaults.responseCharset());
        connections = config.get("connections", defaults.connections(),
            PositiveIntegerValidator.INSTANCE);
        IniConfig timeoutConfig = config.sectionOrNull(TIMEOUT);
        if (timeoutConfig != null) {
            timeout =
                timeoutConfig.getIntegerDuration("socket", defaults.timeout());
            connectTimeout = timeoutConfig.getIntegerDuration(
                "connect",
                defaults.connectTimeout());
            poolTimeout = timeoutConfig.getIntegerDuration(
                "pool",
                defaults.poolTimeout());
            sessionTimeout = timeoutConfig.getIntegerDuration(
                "session",
                defaults.sessionTimeout());
        } else {
            timeout = config.getIntegerDuration(TIMEOUT, defaults.timeout());
            connectTimeout = timeout;
            poolTimeout = timeout;
            sessionTimeout = defaults.sessionTimeout();
        }
        if (config.getString(RETRIES, null) == null) {
            ioRetries = new RetriesConfigBuilder(
                config.section("io-error-retries"),
                defaults.ioRetries());
            httpRetries = new RetriesConfigBuilder(
                config.section("http-error-retries"),
                defaults.ioRetries());
        } else {
            int count = config.get(
                RETRIES,
                defaults.ioRetries().count(),
                NonNegativeIntegerValidator.INSTANCE);
            long interval;
            if (count == 0) {
                interval = 0L;
            } else {
                interval = config.getLongDuration(
                    RETRY_INTERVAL,
                    defaults.ioRetries().interval());
            }
            ioRetries = new RetriesConfigBuilder()
                .count(count)
                .interval(interval);
            count = config.get(
                RETRIES,
                defaults.httpRetries().count(),
                NonNegativeIntegerValidator.INSTANCE);
            if (count == 0) {
                interval = 0L;
            } else {
                interval = config.getLongDuration(
                    RETRY_INTERVAL,
                    defaults.httpRetries().interval());
            }
            httpRetries = new RetriesConfigBuilder()
                .count(count)
                .interval(interval);
        }
        keepAlive = config.getBoolean("keep-alive", defaults.keepAlive());
        contentCompression = config.getBoolean(
            "content-compression",
            defaults.contentCompression());
        redirects = config.getBoolean("redirects", defaults.redirects());
        if (redirects) {
            redirectPosts = config.getBoolean(
                "redirect-posts",
                defaults.redirectPosts());
            relativeRedirects = config.getBoolean(
                "relative-redirects",
                defaults.relativeRedirects());
            circularRedirects = config.getBoolean(
                "circular-redirects",
                defaults.circularRedirects());
            maxRedirects = config.get(
                "max-redirects",
                defaults.maxRedirects(),
                PositiveIntegerValidator.INSTANCE);
        } else {
            redirectPosts = defaults.redirectPosts();
            relativeRedirects = defaults.relativeRedirects();
            circularRedirects = defaults.circularRedirects();
            maxRedirects = defaults.maxRedirects();
        }

        passReferer =
            config.getBoolean("pass-referer", defaults.passReferer());

        httpsConfig = new ClientHttpsConfigBuilder(
            config.section("https"),
            defaults.httpsConfig());

        String dstClientId = config.getString("tvm-client-id", null);

        tvm2Headers =
            config.get(
                "tvm-headers",
                defaults.tvm2Headers(),
                new CollectionParser<>((s) -> {
                    String[] split = s.split(":");
                    return new AbstractMap.SimpleImmutableEntry<>(
                        NonEmptyValidator.TRIMMED.apply(split[0]),
                        NonEmptyValidator.TRIMMED.apply(split[1]));
                },
                ArrayList::new));

        if (dstClientId != null) {
            tvm2Headers = new ArrayList<>(tvm2Headers);
            tvm2Headers.add(
                new AbstractMap.SimpleImmutableEntry<>(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    dstClientId));
        }

        IniConfig statSection = config.sectionOrNull("stat");
        StatersConfig statersDefaults = defaults.statersConfig();
        if (statSection == null) {
            if (statersDefaults == null) {
                this.statersConfig = null;
            } else {
                this.statersConfig = new StatersConfigBuilder(statersDefaults);
            }
        } else if (statersDefaults == null) {
            this.statersConfig = new StatersConfigBuilder(config);
        } else {
            this.statersConfig =
                new StatersConfigBuilder(config, statersDefaults);
        }
    }

    @Override
    public HttpHost proxy() {
        return proxy;
    }

    public T proxy(final HttpHost proxy) {
        this.proxy = proxy;
        return self();
    }

    @Override
    public Charset requestCharset() {
        return requestCharset;
    }

    public T requestCharset(final Charset requestCharset) {
        this.requestCharset = requestCharset;
        return self();
    }

    @Override
    public Charset responseCharset() {
        return responseCharset;
    }

    public T responseCharset(final Charset responseCharset) {
        this.responseCharset = responseCharset;
        return self();
    }

    @Override
    public int connections() {
        return connections;
    }

    public T connections(final int connections) {
        this.connections = connections;
        return self();
    }

    @Override
    public int timeout() {
        return timeout;
    }

    public T timeout(final int timeout) {
        this.timeout = timeout;
        return self();
    }

    @Override
    public int connectTimeout() {
        return connectTimeout;
    }

    public T connectTimeout(final int connectTimeout) {
        this.connectTimeout = connectTimeout;
        return self();
    }

    @Override
    public int poolTimeout() {
        return poolTimeout;
    }

    public T poolTimeout(final int poolTimeout) {
        this.poolTimeout = poolTimeout;
        return self();
    }

    @Override
    public int sessionTimeout() {
        return sessionTimeout;
    }

    public T sessionTimeout(final int sessionTimeout) {
        this.sessionTimeout = sessionTimeout;
        return self();
    }

    @Override
    public RetriesConfigBuilder ioRetries() {
        return ioRetries;
    }

    public T ioRetries(final RetriesConfig ioRetries) {
        this.ioRetries = new RetriesConfigBuilder(ioRetries);
        return self();
    }

    @Override
    public RetriesConfigBuilder httpRetries() {
        return httpRetries;
    }

    public T httpRetries(final RetriesConfig httpRetries) {
        this.httpRetries = new RetriesConfigBuilder(httpRetries);
        return self();
    }

    @Override
    public boolean keepAlive() {
        return keepAlive;
    }

    public T keepAlive(final boolean keepAlive) {
        this.keepAlive = keepAlive;
        return self();
    }

    @Override
    public boolean contentCompression() {
        return contentCompression;
    }

    public T contentCompression(final boolean contentCompression) {
        this.contentCompression = contentCompression;
        return self();
    }

    @Override
    public boolean redirects() {
        return redirects;
    }

    public T redirects(final boolean redirects) {
        this.redirects = redirects;
        return self();
    }

    @Override
    public boolean redirectPosts() {
        return redirectPosts;
    }

    public T redirectPosts(final boolean redirectPosts) {
        this.redirectPosts = redirectPosts;
        return self();
    }

    @Override
    public boolean relativeRedirects() {
        return relativeRedirects;
    }

    public T relativeRedirects(final boolean relativeRedirects) {
        this.relativeRedirects = relativeRedirects;
        return self();
    }

    @Override
    public boolean circularRedirects() {
        return circularRedirects;
    }

    public T circularRedirects(final boolean circularRedirects) {
        this.circularRedirects = circularRedirects;
        return self();
    }

    @Override
    public int maxRedirects() {
        return maxRedirects;
    }

    public T maxRedirects(final int maxRedirects) {
        this.maxRedirects = maxRedirects;
        return self();
    }

    @Override
    public boolean passReferer() {
        return passReferer;
    }

    public T passReferer(final boolean passReferer) {
        this.passReferer = passReferer;
        return self();
    }

    @Override
    public ClientHttpsConfigBuilder httpsConfig() {
        return httpsConfig;
    }

    public T httpsConfig(final ClientHttpsConfig httpsConfig) {
        this.httpsConfig = new ClientHttpsConfigBuilder(httpsConfig);
        return self();
    }

    @Override
    public List<Map.Entry<String, String>> tvm2Headers() {
        return tvm2Headers;
    }

    public T tvm2Headers(final List<Map.Entry<String, String>> headers) {
        this.tvm2Headers = new ArrayList<>(headers);

        return self();
    }

    @Nullable
    @Override
    public StatersConfigBuilder statersConfig() {
        return statersConfig;
    }

    public T statersConfig(final StatersConfig statersConfig) {
        if (statersConfig == null) {
            this.statersConfig = null;
        } else {
            this.statersConfig = new StatersConfigBuilder(statersConfig);
        }
        return self();
    }

    public void copyTo(final AbstractHttpTargetConfigBuilder<?> other) {
        copyTo((AbstractHttpConnectionConfigBuilder<?>) other);
        other
            .proxy(proxy())
            .requestCharset(requestCharset())
            .responseCharset(responseCharset())
            .connections(connections())
            .timeout(timeout())
            .connectTimeout(connectTimeout())
            .poolTimeout(poolTimeout())
            .sessionTimeout(sessionTimeout())
            .ioRetries(ioRetries())
            .httpRetries(httpRetries())
            .keepAlive(keepAlive())
            .contentCompression(contentCompression())
            .redirects(redirects())
            .redirectPosts(redirectPosts())
            .relativeRedirects(relativeRedirects())
            .circularRedirects(circularRedirects())
            .maxRedirects(maxRedirects())
            .passReferer(passReferer())
            .httpsConfig(httpsConfig())
            .tvm2Headers(tvm2Headers())
            .statersConfig(statersConfig());
    }
}

