package ru.yandex.http.config;

import java.io.IOException;
import java.net.Socket;
import java.security.GeneralSecurityException;
import java.security.cert.X509Certificate;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509ExtendedTrustManager;

import ru.yandex.parser.config.ConfigException;

public class ImmutableClientHttpsConfig
    extends ImmutableHttpsConfig
    implements ClientHttpsConfig
{
    private static final TrustManager[] DUMMY_TRUST_MANAGER =
        new TrustManager[] {new DummyTrustManager()};

    private final boolean verifyCertificate;
    private final boolean verifyHostname;
    private final String trustManagerAlgorithm;
    private final SSLContext sslContext;
    private final TrustManagerFactory trustManagerFactory;

    public ImmutableClientHttpsConfig(final ClientHttpsConfig config)
        throws ConfigException
    {
        super(config);
        verifyCertificate =
            VERIFY_CERTIFICATE.validate(config.verifyCertificate());
        verifyHostname = VERIFY_HOSTNAME.validate(config.verifyHostname());
        trustManagerAlgorithm =
            TRUST_MANAGER_ALGORITHM.validate(config.trustManagerAlgorithm());

        try {
            TrustManager[] trustManagers;
            if (verifyCertificate) {
                if (keyStoreConfig() == null) {
                    trustManagers = null;
                    trustManagerFactory = null;
                } else {
                    String jsseProvider = jsseProvider();
                    //TrustManagerFactory trustManagerFactory;
                    if (jsseProvider == null) {
                        trustManagerFactory = TrustManagerFactory.getInstance(
                            trustManagerAlgorithm);
                    } else {
                        trustManagerFactory = TrustManagerFactory.getInstance(
                            trustManagerAlgorithm,
                            jsseProvider);
                    }
                    trustManagerFactory.init(loadKeyStore());
                    trustManagers = trustManagerFactory.getTrustManagers();
                }
            } else {
                trustManagers = DUMMY_TRUST_MANAGER;
                trustManagerFactory = null;
            }

            sslContext = createSSLContext();
            sslContext.init(null, trustManagers, createSecureRandom());
        } catch (GeneralSecurityException | IOException e) {
            throw new ConfigException("Failed to construct SSLContext", e);
        }
    }

    public TrustManagerFactory trustManagerFactory() {
        return trustManagerFactory;
    }

    @Override
    public boolean verifyCertificate() {
        return verifyCertificate;
    }

    @Override
    public boolean verifyHostname() {
        return verifyHostname;
    }

    @Override
    public String trustManagerAlgorithm() {
        return trustManagerAlgorithm;
    }

    public SSLContext getSSLContext() {
        return sslContext;
    }

    private static final class DummyTrustManager
        extends X509ExtendedTrustManager
    {
        private DummyTrustManager() {
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return null;
        }

        @Override
        public void checkClientTrusted(
            final X509Certificate[] certs,
            final String authType)
        {
        }

        @Override
        public void checkClientTrusted(
            final X509Certificate[] certs,
            final String authType,
            final Socket socket)
        {
        }

        @Override
        public void checkClientTrusted(
            final X509Certificate[] certs,
            final String authType,
            final SSLEngine sslEngine)
        {
        }

        @Override
        public void checkServerTrusted(
            final X509Certificate[] certs,
            final String authType)
        {
        }

        @Override
        public void checkServerTrusted(
            final X509Certificate[] certs,
            final String authType,
            final Socket socket)
        {
        }

        @Override
        public void checkServerTrusted(
            final X509Certificate[] certs,
            final String authType,
            final SSLEngine sslEngine)
        {
        }
    }
}

