package ru.yandex.http.config;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import org.apache.http.HttpHost;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.ImmutableStatersConfig;
import ru.yandex.stater.StatersConfig;

public class ImmutableHttpTargetConfig
    extends ImmutableHttpConnectionConfig
    implements HttpTargetConfig
{
    private final HttpHost proxy;
    private final Charset requestCharset;
    private final Charset responseCharset;
    private final int connections;
    private final int timeout;
    private final int connectTimeout;
    private final int poolTimeout;
    private final int sessionTimeout;
    private final ImmutableRetriesConfig ioRetries;
    private final ImmutableRetriesConfig httpRetries;
    private final boolean keepAlive;
    private final boolean contentCompression;
    private final boolean redirects;
    private final boolean redirectPosts;
    private final boolean relativeRedirects;
    private final boolean circularRedirects;
    private final int maxRedirects;
    private final boolean passReferer;
    private final ImmutableClientHttpsConfig httpsConfig;
    private final List<Map.Entry<String, String>> tvm2HeaderClientIds;
    @Nullable
    private final ImmutableStatersConfig statersConfig;

    public ImmutableHttpTargetConfig(final HttpTargetConfig config)
        throws ConfigException
    {
        super(config);
        proxy = config.proxy();
        requestCharset = config.requestCharset();
        responseCharset = config.responseCharset();
        connections = config.connections();
        timeout = config.timeout();
        connectTimeout = config.connectTimeout();
        poolTimeout = config.poolTimeout();
        sessionTimeout = config.sessionTimeout();
        ioRetries = new ImmutableRetriesConfig(config.ioRetries());
        httpRetries =
            new ImmutableRetriesConfig(config.httpRetries());
        keepAlive = config.keepAlive();
        contentCompression = config.contentCompression();
        redirects = config.redirects();
        redirectPosts = config.redirectPosts();
        relativeRedirects = config.relativeRedirects();
        circularRedirects = config.circularRedirects();
        maxRedirects = config.maxRedirects();
        passReferer = config.passReferer();
        httpsConfig = new ImmutableClientHttpsConfig(config.httpsConfig());
        tvm2HeaderClientIds =
            Collections.unmodifiableList(
                new ArrayList<>(config.tvm2Headers()));
        StatersConfig statersConfig = config.statersConfig();
        if (statersConfig == null) {
            this.statersConfig = null;
        } else {
            this.statersConfig = new ImmutableStatersConfig(statersConfig);
        }

        if (connections < 0) {
            throw new ConfigException("Connections count is not set");
        }
        if (maxRedirects <= 0) {
            throw new ConfigException(
                "Max redirects count must be a positive value");
        }
    }

    @Override
    public HttpHost proxy() {
        return proxy;
    }

    @Override
    public Charset requestCharset() {
        return requestCharset;
    }

    @Override
    public Charset responseCharset() {
        return responseCharset;
    }

    @Override
    public int connections() {
        return connections;
    }

    @Override
    public int timeout() {
        return timeout;
    }

    @Override
    public int connectTimeout() {
        return connectTimeout;
    }

    @Override
    public int poolTimeout() {
        return poolTimeout;
    }

    @Override
    public int sessionTimeout() {
        return sessionTimeout;
    }

    @Override
    public ImmutableRetriesConfig ioRetries() {
        return ioRetries;
    }

    @Override
    public ImmutableRetriesConfig httpRetries() {
        return httpRetries;
    }

    @Override
    public boolean keepAlive() {
        return keepAlive;
    }

    @Override
    public boolean contentCompression() {
        return contentCompression;
    }

    @Override
    public boolean redirects() {
        return redirects;
    }

    @Override
    public boolean redirectPosts() {
        return redirectPosts;
    }

    @Override
    public boolean relativeRedirects() {
        return relativeRedirects;
    }

    @Override
    public boolean circularRedirects() {
        return circularRedirects;
    }

    @Override
    public int maxRedirects() {
        return maxRedirects;
    }

    @Override
    public boolean passReferer() {
        return passReferer;
    }

    @Override
    public ImmutableClientHttpsConfig httpsConfig() {
        return httpsConfig;
    }

    @Override
    public List<Map.Entry<String, String>> tvm2Headers() {
        return tvm2HeaderClientIds;
    }

    @Nullable
    @Override
    public ImmutableStatersConfig statersConfig() {
        return statersConfig;
    }
}

