package ru.yandex.http.config;

import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.security.Security;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jsse.provider.BouncyCastleJsseProvider;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.ImmutableConfig;

public class ImmutableHttpsConfig implements HttpsConfig, ImmutableConfig {
    static {
        Security.addProvider(new BouncyCastleProvider());
        Security.addProvider(new BouncyCastleJsseProvider());
    }

    private final String protocol;
    private final List<String> enabledProtocols;
    private final String jsseProvider;
    private final String secureRandomAlgorithm;
    private final String secureRandomProvider;
    private final ImmutableKeyStoreConfig keyStoreConfig;

    public ImmutableHttpsConfig(final HttpsConfig config)
        throws ConfigException
    {
        protocol = PROTOCOL.validate(config.protocol());
        enabledProtocols =
            CollectionCompactor.compact(
                new ArrayList<>(
                    ENABLED_PROTOCOLS.validate(config.enabledProtocols())));
        jsseProvider = JSSE_PROVIDER.validate(config.jsseProvider());
        secureRandomAlgorithm =
            SECURE_RANDOM_ALGORITHM.validate(config.secureRandomAlgorithm());
        secureRandomProvider =
            SECURE_RANDOM_PROVIDER.validate(config.secureRandomProvider());
        keyStoreConfig = KEY_STORE_CONFIG.validate(config.keyStoreConfig());
    }

    @Override
    public String protocol() {
        return protocol;
    }

    @Override
    public List<String> enabledProtocols() {
        return enabledProtocols;
    }

    @Override
    public String jsseProvider() {
        return jsseProvider;
    }

    @Override
    public String secureRandomAlgorithm() {
        return secureRandomAlgorithm;
    }

    @Override
    public String secureRandomProvider() {
        return secureRandomProvider;
    }

    @Override
    public ImmutableKeyStoreConfig keyStoreConfig() {
        return keyStoreConfig;
    }

    public KeyStore loadKeyStore()
        throws GeneralSecurityException, IOException
    {
        KeyStore keyStore = KeyStore.getInstance(keyStoreConfig.type());
        char[] password = keyStoreConfig.password().toCharArray();
        try (InputStream in = keyStoreConfig.file().openStream()) {
            keyStore.load(in, password);
        }
        return keyStore;
    }

    public void initialize(final SSLEngine sslEngine) {
        String[] enabledProtocols;
        if (this.enabledProtocols.isEmpty()) {
            String[] supportedProtocols = sslEngine.getSupportedProtocols();
            List<String> protocols =
                new ArrayList<>(supportedProtocols.length);
            for (String protocol: supportedProtocols) {
                if (!protocol.startsWith("SSL")) {
                    protocols.add(protocol);
                }
            }
            int size = protocols.size();
            if (supportedProtocols.length == size) {
                // No protocols filtered out. Save one allocation
                enabledProtocols = supportedProtocols;
            } else {
                enabledProtocols = protocols.toArray(new String[size]);
            }
        } else {
            enabledProtocols = this.enabledProtocols.toArray(
                new String[this.enabledProtocols.size()]);
        }
        sslEngine.setEnabledProtocols(enabledProtocols);
    }

    public SSLContext createSSLContext() throws GeneralSecurityException {
        if (jsseProvider == null) {
            return SSLContext.getInstance(protocol);
        } else {
            return SSLContext.getInstance(protocol, jsseProvider);
        }
    }

    public SecureRandom createSecureRandom() throws GeneralSecurityException {
        if (secureRandomProvider == null) {
            return SecureRandom.getInstance(secureRandomAlgorithm);
        } else {
            return SecureRandom.getInstance(
                secureRandomAlgorithm,
                secureRandomProvider);
        }
    }
}

