package ru.yandex.http.config;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.Enumeration;
import java.util.Objects;

import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;

import ru.yandex.parser.config.ConfigException;

public class ImmutableServerHttpsConfig
    extends ImmutableHttpsConfig
    implements ServerHttpsConfig
{
    private final String aliasPassword;
    private final int httpPort;
    private final String keyManagerAlgorithm;
    private final SSLContext sslContext;
    private final long certificateValidUntil;

    public ImmutableServerHttpsConfig(final ServerHttpsConfig config)
        throws ConfigException
    {
        super(config);
        aliasPassword = ALIAS_PASSWORD.validate(config.aliasPassword());
        httpPort = HTTP_PORT.validate(config.httpPort());
        keyManagerAlgorithm =
            KEY_MANAGER_ALGORITHM.validate(config.keyManagerAlgorithm());

        if (keyStoreConfig() == null) {
            throw new ConfigException("Key store is not configured");
        }

        try {
            String jsseProvider = jsseProvider();
            KeyManagerFactory keyManagerFactory;
            if (jsseProvider == null) {
                keyManagerFactory =
                    KeyManagerFactory.getInstance(keyManagerAlgorithm);
            } else {
                keyManagerFactory = KeyManagerFactory.getInstance(
                    keyManagerAlgorithm,
                    jsseProvider);
            }
            KeyStore keyStore = loadKeyStore();
            keyManagerFactory.init(
                keyStore,
                Objects.toString(
                    aliasPassword,
                    keyStoreConfig().password())
                    .toCharArray());
            KeyManager[] keyManagers = keyManagerFactory.getKeyManagers();

            sslContext = createSSLContext();
            sslContext.init(keyManagers, null, createSecureRandom());

            Enumeration<String> aliases = keyStore.aliases();
            long minNotAfter = Long.MAX_VALUE;
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                Certificate cert = keyStore.getCertificate(alias);
                if (cert instanceof X509Certificate) {
                    long notAfter =
                        ((X509Certificate) cert).getNotAfter().getTime();
                    if (notAfter < minNotAfter) {
                        minNotAfter = notAfter;
                    }
                }
            }
            certificateValidUntil = minNotAfter;
        } catch (GeneralSecurityException | IOException e) {
            throw new ConfigException("Failed to construct SSLContext", e);
        }
    }

    @Override
    public String aliasPassword() {
        return aliasPassword;
    }

    @Override
    public int httpPort() {
        return httpPort;
    }

    @Override
    public String keyManagerAlgorithm() {
        return keyManagerAlgorithm;
    }

    public SSLContext getSSLContext() {
        return sslContext;
    }

    public long certificateValidUntil() {
        return certificateValidUntil;
    }
}

