package ru.yandex.http.util.nio;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.nio.ContentDecoder;
import org.apache.http.nio.IOControl;

public abstract class AbstractAsyncByteArrayConsumer<T>
    extends AsyncConsumer<T>
{
    private static final byte[] EMPTY_BUF = new byte[0];
    private static final ByteBuffer EMPTY_BB = ByteBuffer.wrap(EMPTY_BUF);
    private static final int DEFAULT_BUFFER_SIZE = 2048;
    private static final long MAX_BUFFER_SIZE = 8192L;

    protected byte[] buf = EMPTY_BUF;
    protected ByteBuffer bb = EMPTY_BB;
    protected int len = 0;
    protected Header contentTypeHeader = null;
    protected Header contentEncodingHeader = null;

    @Override
    public void requestReceived(final HttpRequest request)
        throws HttpException, IOException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            processEntity(((HttpEntityEnclosingRequest) request).getEntity());
        } else {
            processEntity(null);
        }
    }

    @Override
    public void responseReceived(final HttpResponse response)
        throws HttpException, IOException
    {
        processEntity(response.getEntity());
    }

    protected void processEntity(final HttpEntity entity)
        throws HttpException
    {
        if (entity == null) {
            buf = null;
        } else {
            contentTypeHeader = entity.getContentType();
            contentEncodingHeader = entity.getContentEncoding();
            long contentLength = entity.getContentLength();
            if (contentLength == -1L) {
                // This consumer intended for relatively small entities
                // consumption, so there is no need to allocate direct buffer,
                // moreover ChunkDecoder uses SessionInputBuffer for
                // processing, so data will be transferred from off-heap anyway
                bb = ByteBuffer.allocate(DEFAULT_BUFFER_SIZE);
            } else if (contentLength > 0L) {
                if (contentLength <= Integer.MAX_VALUE) {
                    buf = new byte[(int) contentLength];
                }
                bb = ByteBuffer.allocate(
                    (int) Math.min(MAX_BUFFER_SIZE, contentLength));
            }
        }
    }

    @Override
    public void consumeContent(
        final ContentDecoder decoder,
        final IOControl ioctrl)
        throws IOException
    {
        while (true) {
            int read = decoder.read(bb);
            if (read <= 0) {
                break;
            }
            int required = len + read;
            if (required > buf.length) {
                buf = Arrays.copyOf(buf, Math.max(required, buf.length << 1));
            }
            bb.position(0);
            bb.get(buf, len, read);
            len = required;
            bb.clear();
        }
    }
}

