package ru.yandex.http.util.nio;

import java.nio.charset.StandardCharsets;
import java.util.Locale;

import org.apache.http.Header;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;
import org.apache.http.protocol.HTTP;

import ru.yandex.function.VoidProcessor;

public abstract class AbstractEntityGenerator implements EntityGenerator {
    private static final byte[] LF = new byte[]{'\n'};
    private static final int HEADER_OVERHEAD = 3; // ": " + "\n"

    protected final Header contentType;
    protected final Header contentEncoding;

    protected AbstractEntityGenerator(
        final Header contentType,
        final Header contentEncoding)
    {
        this.contentType = contentType;
        this.contentEncoding = contentEncoding;
    }

    protected abstract int expectedBodyLength();

    protected abstract <E extends Exception> void appendBody(
        VoidProcessor<? super byte[], E> processor)
        throws E;

    public static Header contentType(final ContentType contentType) {
        if (contentType == null) {
            return null;
        } else {
            return new BasicHeader(HTTP.CONTENT_TYPE, contentType.toString());
        }
    }

    public AbstractHttpEntity adjustEntity(final AbstractHttpEntity entity) {
        entity.setContentType(contentType);
        entity.setContentEncoding(contentEncoding);
        return entity;
    }

    public static int expectedHeaderStringLength(final Header header) {
        if (header == null) {
            return 0;
        } else {
            return header.getName().length() + header.getValue().length()
                + HEADER_OVERHEAD;
        }
    }

    public static void appendHeader(
        final StringBuilder sb,
        final Header header)
    {
        if (header != null) {
            sb.append(header.getName().toLowerCase(Locale.ROOT));
            sb.append(':');
            sb.append(' ');
            sb.append(header.getValue());
            sb.append('\n');
        }
    }

    @Override
    public int expectedLength() {
        return expectedHeaderStringLength(contentType)
            + expectedHeaderStringLength(contentEncoding)
            + expectedBodyLength();
    }

    protected <E extends Exception> void appendLf(
        final VoidProcessor<? super byte[], E> processor)
        throws E
    {
        processor.process(LF);
    }

    @Override
    public <E extends Exception> void appendTo(
        final VoidProcessor<? super byte[], E> processor)
        throws E
    {
        StringBuilder sb = new StringBuilder(
            expectedHeaderStringLength(contentType)
            + expectedHeaderStringLength(contentEncoding));
        appendHeader(sb, contentType);
        appendHeader(sb, contentEncoding);
        processor.process(new String(sb).getBytes(StandardCharsets.UTF_8));
        appendBody(processor);
    }
}

