package ru.yandex.http.util.nio;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CoderResult;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.nio.ContentDecoder;
import org.apache.http.nio.IOControl;

import ru.yandex.http.util.CharsetUtils;

public abstract class AsyncCharConsumer<T> extends AsyncConsumer<T> {
    private static final int DEFAULT_BUFFER_SIZE = 2048;
    private static final long MAX_BUFFER_SIZE = 8192L;

    protected final long contentLength;
    private final CharsetDecoder decoder;
    private final ByteBuffer bbuf;
    private final CharBuffer cbuf;

    protected AsyncCharConsumer(final HttpEntity entity) throws HttpException {
        contentLength = contentLength(entity);
        decoder = CharsetUtils.decoder(entity);
        int bufferSize;
        if (contentLength == -1L) {
            bufferSize = DEFAULT_BUFFER_SIZE;
        } else {
            bufferSize = (int) Math.min(MAX_BUFFER_SIZE, contentLength);
        }
        bbuf = ByteBuffer.allocate(bufferSize);
        cbuf = CharBuffer.allocate(bufferSize);
    }

    public static long contentLength(final HttpEntity entity) {
        if (entity == null) {
            return -1L;
        } else {
            return entity.getContentLength();
        }
    }

    protected abstract void consumeContent(
        final CharBuffer cbuf,
        final IOControl ioctrl)
        throws IOException;

    @Override
    public void consumeContent(
        final ContentDecoder decoder,
        final IOControl ioctrl)
        throws IOException
    {
        while (true) {
            int read = decoder.read(bbuf);
            boolean completed = decoder.isCompleted();
            if (read == 0 && !completed) {
                break;
            }
            if (read == -1) {
                // decored.isCompleted() was already set at previous read
                break;
            }
            bbuf.flip();
            handleCoderResult(
                this.decoder.decode(bbuf, cbuf, completed),
                ioctrl);
            bbuf.compact();
            if (completed) {
                handleCoderResult(this.decoder.flush(cbuf), ioctrl);
                break;
            }
        }
    }

    private void handleCoderResult(
        final CoderResult result,
        final IOControl ioctrl)
        throws IOException
    {
        if (result.isError()) {
            result.throwException();
        }
        cbuf.flip();
        if (cbuf.hasRemaining()) {
            consumeContent(cbuf, ioctrl);
        }
        cbuf.clear();
    }
}

