package ru.yandex.http.util.nio;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncResponseConsumer;
import org.apache.http.protocol.HttpContext;

public abstract class AsyncConsumer<T>
    implements HttpAsyncRequestConsumer<T>, HttpAsyncResponseConsumer<T>
{
    private final AtomicBoolean completed = new AtomicBoolean();
    private volatile T result = null;
    private volatile Exception e = null;

    protected abstract T buildResult(final HttpContext context)
        throws Exception;

    protected void releaseResources() {
    }

    protected void onClose() throws IOException {
    }

    @Override
    public void requestReceived(final HttpRequest request)
        throws HttpException, IOException
    {
    }

    @Override
    public void responseReceived(final HttpResponse response)
        throws HttpException, IOException
    {
    }

    @Override
    public void requestCompleted(final HttpContext context) {
        completed(context);
    }

    @Override
    public void responseCompleted(final HttpContext context) {
        completed(context);
    }

    private void completed(final HttpContext context) {
        if (completed.compareAndSet(false, true)) {
            try {
                result = buildResult(context);
            } catch (Exception e) {
                this.e = e;
            } finally {
                releaseResources();
            }
        }
    }

    @Override
    public boolean cancel() {
        if (completed.compareAndSet(false, true)) {
            releaseResources();
            return true;
        }
        return false;
    }

    @Override
    public void failed(final Exception e) {
        if (completed.compareAndSet(false, true)) {
            this.e = e;
            releaseResources();
        }
    }

    @Override
    public void close() throws IOException {
        if (completed.compareAndSet(false, true)) {
            releaseResources();
            onClose();
        }
    }

    @Override
    public Exception getException() {
        return e;
    }

    @Override
    public T getResult() {
        return result;
    }

    @Override
    public boolean isDone() {
        return completed.get();
    }
}

