package ru.yandex.http.util.nio;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpHost;
import org.apache.http.HttpMessage;
import org.apache.http.HttpRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHttpEntityEnclosingRequest;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.function.Processable;
import ru.yandex.function.VoidProcessor;
import ru.yandex.http.util.HeaderUtils;

public class BasicAsyncRequestProducerGenerator
    implements Function<HttpHost, BasicAsyncRequestProducer>
{
    private final String uri;
    private final EntityGenerator entityGenerator;
    private final String method;
    private List<Header> headers = null;

    public BasicAsyncRequestProducerGenerator(final String uri) {
        this(uri, null, "GET");
    }

    public BasicAsyncRequestProducerGenerator(
        final String uri,
        final String content)
    {
        this(uri, content, StandardCharsets.UTF_8);
    }

    public BasicAsyncRequestProducerGenerator(
        final String uri,
        final String content,
        final Charset charset)
    {
        this(uri, content, ContentType.TEXT_PLAIN.withCharset(charset));
    }

    public BasicAsyncRequestProducerGenerator(
        final String uri,
        final String content,
        final ContentType contentType)
    {
        this(uri, new NByteArrayEntityGenerator(content, contentType));
    }

    public BasicAsyncRequestProducerGenerator(
        final String uri,
        final byte[] content,
        final ContentType contentType)
    {
        this(uri, new NByteArrayEntityGenerator(content, contentType));
    }

    public BasicAsyncRequestProducerGenerator(
        final String uri,
        final Processable<byte[]> content,
        final ContentType contentType)
    {
        this(uri, new NByteArrayEntityGenerator(content, contentType));
    }

    public BasicAsyncRequestProducerGenerator(
        final String uri,
        final HttpEntity entity)
        throws IOException
    {
        this(uri, new NByteArrayEntityGenerator(entity));
    }

    public BasicAsyncRequestProducerGenerator(
        final String uri,
        final EntityGenerator entityGenerator)
    {
        this(uri, entityGenerator, "POST");
    }

    public BasicAsyncRequestProducerGenerator(
        final String uri,
        final EntityGenerator entityGenerator,
        final String method)
    {
        this.uri = uri;
        if (entityGenerator == EmptyNHttpEntity.INSTANCE) {
            this.entityGenerator = null;
        } else {
            this.entityGenerator = entityGenerator;
        }
        this.method = method;
    }

    public BasicAsyncRequestProducerGenerator(final HttpRequest request)
        throws IOException
    {
        this(request.getRequestLine().getUri(), request);
    }

    public BasicAsyncRequestProducerGenerator(
        final String uri,
        final HttpRequest request)
        throws IOException
    {
        this.uri = uri;
        if (request instanceof HttpEntityEnclosingRequest) {
            entityGenerator = new NByteArrayEntityGenerator(
                ((HttpEntityEnclosingRequest) request).getEntity());
        } else {
            entityGenerator = null;
        }
        method = request.getRequestLine().getMethod();
    }

    public BasicAsyncRequestProducerGenerator(
        final BasicAsyncRequestProducerGenerator sample)
    {
        uri = sample.uri;
        entityGenerator = sample.entityGenerator;
        method = sample.method;
        if (sample.headers != null) {
            headers = new ArrayList<>(sample.headers);
        }
    }

    public void addHeader(final String name, final String value) {
        addHeader(HeaderUtils.createHeader(name, value));
    }

    public void addHeader(final Header header) {
        if (headers == null) {
            headers = new ArrayList<>(1);
        }
        headers.add(header);
    }

    public void copyHeader(final HttpMessage source, final String headerName) {
        Header header = source.getFirstHeader(headerName);
        if (header != null) {
            addHeader(header);
        }
    }

    @Override
    public BasicAsyncRequestProducer apply(final HttpHost host) {
        HttpRequest request;
        if (entityGenerator == null) {
            request = new BasicHttpRequest(method, uri);
        } else {
            BasicHttpEntityEnclosingRequest enclosingRequest =
                new BasicHttpEntityEnclosingRequest(method, uri);
            enclosingRequest.setEntity(entityGenerator.get());
            request = enclosingRequest;
        }
        if (headers != null) {
            for (Header header: headers) {
                request.addHeader(header);
            }
        }
        return new BasicAsyncRequestProducer(host, request);
    }

    @Override
    public String toString() {
        if (headers == null) {
            return uri;
        } else {
            return uri + ' ' + headers;
        }
    }

    private int expectedHeadLength(
        final Predicate<? super Header> headersFilter)
    {
        int expectedLength = method.length() + uri.length() + 2;
        if (headers != null) {
            for (Header header: headers) {
                if (headersFilter.test(header)) {
                    expectedLength +=
                        AbstractEntityGenerator.expectedHeaderStringLength(
                            header);
                }
            }
        }
        return expectedLength;
    }

    public int expectedLength(final Predicate<? super Header> headersFilter) {
        int expectedLength = expectedHeadLength(headersFilter);
        if (entityGenerator != null) {
            expectedLength += entityGenerator.expectedLength();
        }
        return expectedLength;
    }

    private void appendHead(
        final StringBuilder sb,
        final Predicate<? super Header> headersFilter)
    {
        sb.append(method);
        sb.append(' ');
        sb.append(uri);
        sb.append('\n');
        if (headers != null) {
            for (Header header: headers) {
                if (headersFilter.test(header)) {
                    AbstractEntityGenerator.appendHeader(sb, header);
                }
            }
        }
    }

    public <E extends Exception> void appendTo(
        final VoidProcessor<? super byte[], E> processor,
        final Predicate<? super Header> headersFilter)
        throws E
    {
        StringBuilder sb =
            new StringBuilder(expectedHeadLength(headersFilter));
        appendHead(sb, headersFilter);
        processor.process(new String(sb).getBytes(StandardCharsets.UTF_8));
        if (entityGenerator != null) {
            entityGenerator.appendTo(processor);
        }
    }
}

