package ru.yandex.http.util.nio;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.apache.http.Header;
import org.apache.http.HeaderIterator;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpVersion;
import org.apache.http.StatusLine;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHttpResponse;
import org.apache.http.message.BasicStatusLine;
import org.apache.http.protocol.HTTP;

import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.YandexReasonPhraseCatalog;

public class BasicAsyncResponseProducerGenerator {
    private final StatusLine statusLine;
    private final EntityGenerator entityGenerator;
    private List<Header> headers = null;

    public BasicAsyncResponseProducerGenerator(final int httpCode) {
        this(httpCode, (EntityGenerator) null);
    }

    public BasicAsyncResponseProducerGenerator(
        final int httpCode,
        final String content)
    {
        this(httpCode, content, StandardCharsets.UTF_8);
    }

    public BasicAsyncResponseProducerGenerator(
        final int httpCode,
        final String content,
        final Charset charset)
    {
        this(httpCode, content, ContentType.TEXT_PLAIN.withCharset(charset));
    }

    public BasicAsyncResponseProducerGenerator(
        final int httpCode,
        final String content,
        final ContentType contentType)
    {
        this(httpCode, new NByteArrayEntityGenerator(content, contentType));
    }

    public BasicAsyncResponseProducerGenerator(
        final int httpCode,
        final byte[] content,
        final ContentType contentType)
    {
        this(httpCode, new NByteArrayEntityGenerator(content, contentType));
    }

    public BasicAsyncResponseProducerGenerator(
        final int httpCode,
        final HttpEntity entity)
        throws IOException
    {
        this(httpCode, new NByteArrayEntityGenerator(entity));
    }

    public BasicAsyncResponseProducerGenerator(
        final int httpCode,
        final EntityGenerator entityGenerator)
    {
        this.statusLine = new BasicStatusLine(
            HttpVersion.HTTP_1_1,
            httpCode,
            YandexReasonPhraseCatalog.INSTANCE.getReason(
                httpCode,
                Locale.ENGLISH));
        this.entityGenerator = entityGenerator;
    }

    public BasicAsyncResponseProducerGenerator(final HttpResponse response)
        throws IOException
    {
        this(response.getStatusLine(), response);
    }

    public BasicAsyncResponseProducerGenerator(
        final StatusLine statusLine,
        final HttpResponse response)
        throws IOException
    {
        this.statusLine = statusLine;
        if (response.getEntity() != null) {
            entityGenerator =
                new NByteArrayEntityGenerator(response.getEntity());
        } else {
            entityGenerator = null;
        }
        headers = new ArrayList<>(1 << (2 + 1));
        HeaderIterator headerIterator = response.headerIterator();
        while (headerIterator.hasNext()) {
            Header header = headerIterator.nextHeader();
            String name = header.getName();
            if (name.equalsIgnoreCase(HTTP.CONTENT_LEN)
                || name.equalsIgnoreCase(HTTP.CONTENT_TYPE)
                || name.equalsIgnoreCase(HTTP.CONTENT_ENCODING)
                || name.equalsIgnoreCase(HTTP.CONN_DIRECTIVE)
                || name.equalsIgnoreCase(HTTP.CONN_KEEP_ALIVE)
                || name.equalsIgnoreCase(HTTP.TRANSFER_ENCODING))
            {
                continue;
            }
            headers.add(header);
        }
    }

    public void addHeader(final String name, final String value) {
        addHeader(HeaderUtils.createHeader(name, value));
    }

    public void addHeader(final Header header) {
        if (headers == null) {
            headers = new ArrayList<>(1);
        }
        headers.add(header);
    }

    private HttpResponse prepareResponse() {
        HttpResponse response = new BasicHttpResponse(statusLine);
        if (entityGenerator != null) {
            response.setEntity(entityGenerator.get());
        }
        if (headers != null) {
            for (Header header: headers) {
                response.addHeader(header);
            }
        }
        return response;
    }

    public BasicAsyncResponseProducer get() {
        return new BasicAsyncResponseProducer(prepareResponse());
    }

    public BasicAsyncResponseProducer get(final Header additionalHeader) {
        HttpResponse response = prepareResponse();
        response.addHeader(additionalHeader);
        return new BasicAsyncResponseProducer(response);
    }

    public BasicAsyncResponseProducer get(
        final Header additionalHeader1,
        final Header additionalHeader2)
    {
        HttpResponse response = prepareResponse();
        response.addHeader(additionalHeader1);
        response.addHeader(additionalHeader2);
        return new BasicAsyncResponseProducer(response);
    }

    public BasicAsyncResponseProducer get(final Header... additionalHeaders) {
        HttpResponse response = prepareResponse();
        for (Header header: additionalHeaders) {
            response.addHeader(header);
        }
        return new BasicAsyncResponseProducer(response);
    }

    public List<Header> headers() {
        return headers;
    }

    public EntityGenerator entityGenerator() {
        return entityGenerator;
    }

    @Override
    public String toString() {
        if (headers == null) {
            return statusLine.toString();
        } else {
            return statusLine.toString() + ' ' + headers;
        }
    }
}

