package ru.yandex.http.util.nio;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;

import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.Processable;
import ru.yandex.function.VoidProcessor;
import ru.yandex.http.util.CharsetUtils;

public class NByteArrayEntityGenerator extends AbstractEntityGenerator {
    protected final Processable<byte[]> content;

    public NByteArrayEntityGenerator(
        final byte[] content,
        final ContentType contentType)
    {
        this(content, contentType, null);
    }

    public NByteArrayEntityGenerator(
        final byte[] content,
        final ContentType contentType,
        final Header contentEncoding)
    {
        this(content, contentType(contentType), contentEncoding);
    }

    public NByteArrayEntityGenerator(
        final byte[] content,
        final Header contentType,
        final Header contentEncoding)
    {
        this(new ByteArrayProcessable(content), contentType, contentEncoding);
    }

    public NByteArrayEntityGenerator(final HttpEntity entity)
        throws IOException
    {
        this(
            CharsetUtils.toDecodable(entity),
            entity.getContentType(),
            entity.getContentEncoding());
    }

    public NByteArrayEntityGenerator(
        final Processable<byte[]> content,
        final ContentType contentType)
    {
        this(content, contentType, null);
    }

    public NByteArrayEntityGenerator(
        final Processable<byte[]> content,
        final ContentType contentType,
        final Header contentEncoding)
    {
        this(
            content,
            contentType(contentType),
            contentEncoding);
    }

    public NByteArrayEntityGenerator(
        final String content,
        final ContentType contentType)
    {
        this(content, contentType, null);
    }

    public NByteArrayEntityGenerator(
        final String content,
        final ContentType contentType,
        final Header contentEncoding)
    {
        this(
            content.getBytes(charset(contentType)),
            contentType,
            contentEncoding);
    }

    protected NByteArrayEntityGenerator(
        final Processable<byte[]> content,
        final Header contentType,
        final Header contentEncoding)
    {
        super(contentType, contentEncoding);
        this.content = content;
    }

    @Override
    public HttpEntity get() {
        return adjustEntity(
            content.processWith(NByteArrayEntityFactory.INSTANCE));
    }

    @Override
    protected int expectedBodyLength() {
        if (content == null) {
            return 0;
        } else {
            return 1 + content.length();
        }
    }

    @Override
    protected <E extends Exception> void appendBody(
        final VoidProcessor<? super byte[], E> processor)
        throws E
    {
        if (content != null) {
            appendLf(processor);
            content.processWith(processor);
        }
    }

    private static Charset charset(final ContentType contentType) {
        Charset charset;
        if (contentType == null) {
            charset = StandardCharsets.UTF_8;
        } else {
            charset = contentType.getCharset();
            if (charset == null) {
                charset = StandardCharsets.UTF_8;
            }
        }
        return charset;
    }
}

