package ru.yandex.http.proxy;

import java.nio.charset.Charset;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.Cancellable;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncResponseProducer;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.collection.IdentityHashSet;
import ru.yandex.http.server.async.BaseAsyncServer;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HeadersParser;
import ru.yandex.http.util.nio.client.BasicRequestsListener;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.OpenApiCgiParams;
import ru.yandex.parser.uri.OpenApiDocumentation;
import ru.yandex.parser.uri.UriParser;

public class BasicProxySession
    extends UriParser
    implements ProxySession, Cancellable
{
    private static final CollectionParser<
        String,
        Set<String>,
        Exception>
        DEBUG_FLAGS_PARSER =
            new CollectionParser<>(
                NonEmptyValidator.INSTANCE.compose(String::intern),
                IdentityHashSet::new);

    private final RequestsListener listener = new BasicRequestsListener();
    private final BaseAsyncServer<?> server;
    private final HttpAsyncExchange exchange;
    protected final HttpCoreContext context;
    protected final HttpRequest request;
    private final Charset acceptedCharset;
    private final CgiParams params;
    private final Set<String> debugFlags;
    protected final HeadersParser headers;
    private final PrefixedLogger logger;
    protected final LoggingServerConnection connection;
    private Set<Cancellable> cancellables;
    private volatile boolean cancelled = false;

    public BasicProxySession(
        final BaseAsyncServer<?> server,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        this(server, exchange, HttpCoreContext.adapt(context));
    }

    public BasicProxySession(
        final BaseAsyncServer<?> server,
        final HttpAsyncExchange exchange,
        final HttpCoreContext context)
        throws HttpException
    {
        super(exchange.getRequest().getRequestLine().getUri());
        this.server = server;
        this.exchange = exchange;
        this.context = context;
        request = exchange.getRequest();
        acceptedCharset = CharsetUtils.acceptedCharset(request);
        if (OpenApiDocumentation.enabled()) {
            // handle manual BasicProxySession creation
            if (!(this instanceof OpenApiProxySession)) {
                this.params = new OpenApiCgiParams(
                    new BasicHttpRequest(request.getRequestLine()), queryParser());
            } else {
                params = new CgiParams(queryParser());
            }
        } else {
            params = new CgiParams(queryParser());
        }

        debugFlags = params.<Set<String>>get(
            "debug-flags",
            server.debugFlags(),
            DEBUG_FLAGS_PARSER);
        headers = new HeadersParser(request);
        logger = context.getAttribute(HttpProxy.LOGGER, PrefixedLogger.class);
        connection = context.getConnection(LoggingServerConnection.class);
        connection.setUpstreamStats(listener);
    }

    @Override
    public Charset acceptedCharset() {
        return acceptedCharset;
    }

    @Override
    public HttpRequest request() {
        return request;
    }

    @Override
    public UriParser uri() {
        return this;
    }

    @Override
    public CgiParams params() {
        return params;
    }

    @Override
    public HeadersParser headers() {
        return headers;
    }

    @Override
    public HttpCoreContext context() {
        return context;
    }

    @Override
    public HttpAsyncExchange exchange() {
        return exchange;
    }

    @Override
    public PrefixedLogger logger() {
        return logger;
    }

    @Override
    public RequestsListener listener() {
        return listener;
    }

    @Override
    public LoggingServerConnection connection() {
        return connection;
    }

    @Override
    public boolean cancelled() {
        return cancelled;
    }

    @Override
    public Set<String> debugFlags() {
        return debugFlags;
    }

    @Override
    public synchronized boolean cancel() {
        if (!cancelled) {
            cancelled = true;
            boolean success = true;
            if (cancellables != null) {
                Iterator<Cancellable> iter = cancellables.iterator();
                while (iter.hasNext()) {
                    Cancellable callback = iter.next();
                    try {
                        if (!callback.cancel()) {
                            success = false;
                        }
                    } catch (Throwable t) {
                        logger.log(
                            Level.SEVERE,
                            "cancel(): error calling cancel",
                            t);
                    }
                    iter.remove();
                }
            }
            return success;
        } else {
            return true;
        }
    }

    @Override
    public synchronized void subscribeForCancellation(
        final Cancellable callback)
    {
        if (cancelled) {
            callback.cancel();
            return;
        }
        if (cancellables == null) {
            cancellables = new LinkedHashSet<>();
            exchange.setCallback(this);
        }
        cancellables.add(callback);
    }

    @Override
    public synchronized void unsubscribeFromCancellation(
        final Cancellable callback)
    {
        cancellables.remove(callback);
        if (cancellables.size() == 0) {
            cancellables = null;
        }
    }

    @Override
    public HttpResponse getResponse() {
        return exchange.getResponse();
    }

    @Override
    public void response(final int status, final HttpEntity entity) {
        exchange.getResponse().setStatusCode(status);
        if (entity != null) {
            exchange.getResponse().setEntity(entity);
        }
        exchange.submitResponse();
        cancel();
    }

    @Override
    public void response(final HttpAsyncResponseProducer producer) {
        exchange.submitResponse(producer);
        cancel();
    }

    @Override
    public void handleException(final HttpException e) {
        exchange.submitResponse(server.handleException(e, context));
        cancel();
    }
}

