package ru.yandex.http.proxy;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Function;

import org.apache.http.nio.protocol.HttpAsyncRequestHandler;

import ru.yandex.collection.Pattern;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.server.async.BaseAsyncServer;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncClientRegistrar;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.ImmutableHttpProxyConfig;
import ru.yandex.parser.uri.OpenApiDocumentation;
import ru.yandex.stater.ImmutableStatersConfig;

public class HttpProxy<C extends ImmutableHttpProxyConfig>
    extends BaseAsyncServer<C>
    implements AsyncClientRegistrar
{
    protected final Map<String, AbstractAsyncClient<?>> clients =
        new LinkedHashMap<>();

    protected final SharedConnectingIOReactor reactor;
    protected final MultiClientTvm2TicketSupplier tvm2ClientTicketGenerator;

    public HttpProxy(final C config) throws IOException {
        super(config);
        reactor = new SharedConnectingIOReactor(
            config,
            config.dnsConfig(),
            new ThreadGroup(getThreadGroup(), config.name() + "-C"));
        closeChain.add(reactor);
        if (config.tvm2ServiceConfig() != null) {
            tvm2ClientTicketGenerator = new MultiClientTvm2TicketSupplier(logger(), serviceContextRenewalTask);
            closeChain.add(tvm2ClientTicketGenerator);
        } else {
            tvm2ClientTicketGenerator = null;
        }
    }

    @Override
    public SharedConnectingIOReactor reactor() {
        return reactor;
    }

    @Override
    public void start() throws IOException {
        reactor.start();
        if (tvm2ClientTicketGenerator != null) {
            tvm2ClientTicketGenerator.start();
        }

        clients.values().forEach(client -> client.start());
        super.start();
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put("dns_resolver", reactor.dnsResolver().status(verbose));
        clients.forEach(
            (name, client) -> status.put(name, client.status(verbose)));
        return status;
    }

    public void register(
        final Pattern<RequestInfo> pattern,
        final ProxyRequestHandler handler)
    {
        register(pattern, proxyHandlerAdapter(handler));
    }

    public void register(
        final Pattern<RequestInfo> pattern,
        final ProxyRequestHandler handler,
        final String... methods)
    {
        register(
            pattern,
            proxyHandlerAdapter(handler),
            methods);
    }

    public HttpAsyncRequestHandler<?> register(
        final Pattern<RequestInfo> pattern,
        final ProxyRequestHandler handler,
        final String method)
    {
        return register(
            pattern,
            proxyHandlerAdapter(handler),
            method);
    }

    private ProxyRequestHandlerAdapter proxyHandlerAdapter(final ProxyRequestHandler handler) {
        if (OpenApiDocumentation.enabled()) {
            return new OpenApiProxyRequestHandlerAdapter(handler, this);
        } else {
            return new ProxyRequestHandlerAdapter(handler, this);
        }
    }

    @Override
    public AsyncClient client(
        final String name,
        final ImmutableHttpTargetConfig config,
        final Function<? super Exception, RequestErrorType> errorClassifier)
    {
        AsyncClient client = new AsyncClient(reactor, config, errorClassifier);
        return registerClient(name, client, config);
    }

    @Override
    public <T extends AbstractAsyncClient<T>> T registerClient(
        final String name,
        final T client,
        final ImmutableHttpTargetConfig config)
    {
        closeChain.add(client);
        AbstractAsyncClient<?> oldClient = clients.put(name, client);
        if (oldClient != null) {
            throw new IllegalStateException(
                "Can't register client " + client
                + " with name " + name
                + " because there is already client " + oldClient);
        }
        ImmutableStatersConfig statersConfig = client.statersConfig();
        if (statersConfig != null) {
            registerStaters(statersConfig);
        }
        if (tvm2ClientTicketGenerator == null
            || config.tvm2Headers().isEmpty())
        {
            logger().info(
                "No tvm headers configured for client " + name);
            return client;
        } else {
            logger().info(
                "Registering tvm headers for client " + name
                + ": " + config.tvm2Headers());
            return client.addHeader(
                new HttpProxyTvm2HeaderSupplier(
                    tvm2ClientTicketGenerator,
                    config.tvm2Headers()));
        }
    }
}

